<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Domain;

use App\Operation\ExecuteShellCommand;
use Exception;
use Symfony\Component\Filesystem\Filesystem;

final class XCart
{
    private string $sourcePath;

    private ExecuteShellCommand $executeShellCommand;

    private Filesystem $filesystem;

    public function __construct(
        string $xcartSourcePath,
        ExecuteShellCommand $executeShellCommand,
        Filesystem $filesystem
    ) {
        $this->sourcePath          = rtrim(realpath($xcartSourcePath), '/') . '/';
        $this->executeShellCommand = $executeShellCommand;
        $this->filesystem          = $filesystem;
    }

    public function getCodeSourcePath(): string
    {
        return $this->sourcePath . 'classes/';
    }

    public function getCodeCachePath(): string
    {
        return $this->sourcePath . 'var/run/classes/';
    }

    public function getModulesPath(): string
    {
        return $this->sourcePath . 'modules/';
    }

    public function getPacksPath(): string
    {
        return $this->sourcePath . 'var/packs/';
    }

    public function getGmvPath(): string
    {
        return $this->sourcePath . 'var/gmv/';
    }

    public function run(string $command, array $params = []): string
    {
        return ($this->executeShellCommand)(['php', './bin/console', ...explode(' ', $command), ...$params], $this->sourcePath);
    }

    public function runComposer(string $command, array $params = []): string
    {
        return ($this->executeShellCommand)(['composer', "--working-dir={$this->sourcePath}", $command, ...$params]);
    }

    public function writeToFile(string $fileName, string $content): bool
    {
        $filePath = $this->sourcePath . $fileName;
        $this->filesystem->mkdir(dirname($filePath));

        return (bool) file_put_contents($filePath, $content);
    }

    public function getSourcePath(): string
    {
        return $this->sourcePath;
    }

    public function setSourcePath(string $sourcePath): void
    {
        $this->sourcePath = $sourcePath;
    }

    /**
     * @throws Exception
     */
    public function getCoreVersion(): string
    {
        preg_match(
            "/const XC_VERSION = '(\d+.\d+.\d+.\d+)';/",
            file_get_contents($this->sourcePath . 'classes/XLite.php'),
            $coreVersion
        );

        if (isset($coreVersion[1])) {
            return $coreVersion[1];
        } else {
            throw new Exception('Core Version is not found');
        }
    }
}
