<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Domain;

use App\Entity\Module;
use App\Exception\ModuleDomainException;
use SplFileInfo;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Finder\Finder;
use Symfony\Component\Yaml\Yaml;
use Exception;

final class ModuleDomain
{
    private string $sourcePath;

    private Filesystem $filesystem;

    public function __construct(
        XCart $XCart,
        Filesystem $filesystem
    ) {
        $this->filesystem = $filesystem;

        $this->setSourcePath($XCart->getSourcePath());
    }

    /**
     * @throws Exception
     */
    public function readModuleInfo(string $module): ?array
    {
        try {
            [$author, $name] = explode('-', $module);
        } catch (Exception $e) {
            throw ModuleDomainException::fromIncorrectModuleIdFormat($module);
        }

        $mainYamlPath = $this->getModulesCodePath() . "$author/$name/config/main.yaml";

        if (!is_readable($mainYamlPath)) {
            throw ModuleDomainException::fromNoConfigFile($mainYamlPath);
        }

        $moduleData = Yaml::parseFile($mainYamlPath);

        return [
            'author'   => $author,
            'name'     => $name,
            'source'   => Module::SOURCE_LOCAL,
            'version'  => $moduleData['version'] ?? '5.5.0.0',
            'state'    => Module::STATE_NOT_INSTALLED,
            'metaData' => [
                'type'                     => $moduleData['type'] ?? 'common',
                'authorName'               => $moduleData['authorName'] ?? $author,
                'moduleName'               => $moduleData['moduleName'] ?? $name,
                'description'              => $moduleData['description'] ?? '',
                'minorRequiredCoreVersion' => $moduleData['minorRequiredCoreVersion'] ?? 0,
                'dependsOn'                => $moduleData['dependsOn'] ?? [],
                'incompatibleWith'         => $moduleData['incompatibleWith'] ?? [],
                'showSettingsForm'         => $moduleData['showSettingsForm'] ?? false,
                'isSystem'                 => $moduleData['isSystem'] ?? false,
                'canDisable'               => $moduleData['canDisable'] ?? true,
                'autoloader'               => $moduleData['autoloader'] ?? [],
            ],
        ];
    }

    public function searchLocalModules(): array
    {
        $modules         = [];
        $modulesCodePath = $this->getModulesCodePath();

        if ($this->filesystem->exists($modulesCodePath)) {
            $finder = new Finder();
            $finder->files()
                ->in($this->getModulesCodePath())
                ->depth('== 3')
                ->name('main.yaml');

            foreach ($finder as $file) {
                $modules[] = str_replace('/', '-', dirname($file->getRelativePath()));
            }
        }

        return $modules;
    }

    public function searchModuleInstallFixtures(string $moduleId): array
    {
        if ($moduleId === 'CDev-Core') {
            return [
                $this->getSourcePath() . 'sql/xlite_data.yaml',
                $this->getSourcePath() . 'sql/xlite_data_lng.yaml',
            ];
        }

        [$author, $name] = explode('-', $moduleId);

        $modulePath = $this->getModulesCodePath() . $author . '/' . $name . '/';
        if (!is_dir($modulePath) || !is_readable($modulePath)) {
            return [];
        }

        $finder = new Finder();
        $finder->files()
            ->in($this->getModulesCodePath() . $author . '/' . $name . '/config')
            ->depth('== 0')
            ->name('install*.yaml')
            ->sort(static function (SplFileInfo $a, SplFileInfo $b) {
                if ($a->getFilename() === 'install.yaml') {
                    return -1;
                }

                if ($b->getFilename() === 'install.yaml') {
                    return 1;
                }

                return strcmp($a->getFilename(), $b->getFilename());
            });

        $files = [];
        foreach ($finder as $file) {
            $files[] = (string) $file;
        }

        return $files;
    }

    public function getModulesCodePath(): string
    {
        return $this->sourcePath . 'modules/';
    }

    public function getSourcePath(): string
    {
        return $this->sourcePath;
    }

    public function setSourcePath(string $sourcePath): void
    {
        $this->sourcePath = rtrim($sourcePath, '/') . '/';
    }
}
