<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Domain;

use App\Exception\HookManagerException;

final class HookManagerDomain
{
    public const HOOK_TYPE_UPGRADE = 'upgrade';

    private array $hooks = [];

    public function addHook(array $hook): void
    {
        $object   = $hook['object'] ?? null;
        $hookType = $hook['hookType'] ?? self::HOOK_TYPE_UPGRADE;

        $this->hooks[$hookType][] = [
            'object'  => $object,
            'method'  => $hook['method'],
            'version' => $hook['version'] ?? '',
        ];
    }

    /**
     * @throws HookManagerException
     */
    public function runHook(array $hook): int
    {
        $hookType = $hook['hookType'] ?? '';

        $result = 0;

        try {
            switch ($hookType) {
                case self::HOOK_TYPE_UPGRADE:
                    $result = $this->runUpgradeHook($hook['versionFrom'], $hook['versionTo']);
                    break;
            }
        } catch (\Exception $e) {
            throw HookManagerException::fromRunHook($e, $hookType);
        }

        return $result;
    }

    private function runUpgradeHook(string $versionFrom, string $versionTo): int
    {
        if (isset($this->hooks[self::HOOK_TYPE_UPGRADE])) {
            $upgradeHooks = array_filter(
                $this->hooks[self::HOOK_TYPE_UPGRADE],
                static fn ($upgradeHook) => version_compare($upgradeHook['version'], $versionFrom, '>') && version_compare($upgradeHook['version'], $versionTo, '<=')
            );

            usort($upgradeHooks, [$this, 'sortByVersion']);

            foreach ($upgradeHooks as $upgradeHook) {
                $object = $upgradeHook['object'];
                $method = $upgradeHook['method'];

                $object->$method();
            }

            return count($upgradeHooks);
        }

        return 0;
    }

    private function sortByVersion($hook1, $hook2): int
    {
        return version_compare($hook1['version'], $hook2['version']);
    }

    /**
     * @return array
     */
    public function getHooks(): array
    {
        return $this->hooks;
    }

    /**
     * @param array $hooks
     */
    public function setHooks(array $hooks): void
    {
        $this->hooks = $hooks;
    }
}
