<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Deployment\Step;

use App\Entity\Scenario;
use App\Exception\StepNotReadyException;
use LogicException;
use SplFileInfo;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Finder\Finder;

final class WaitForWorkersStop extends Step
{
    private Filesystem $filesystem;

    private int $sleepForSec;

    private int $maxSleepTimeSec;

    private Finder $workers;

    public function __construct(
        Filesystem $filesystem,
        string $workersPoolPath,
        int $sleepForSec,
        int $maxSleepTimeSec
    ) {
        $this->filesystem = $filesystem;
        $this->sleepForSec = $sleepForSec;
        $this->maxSleepTimeSec = $maxSleepTimeSec;

        $this->filesystem->mkdir($workersPoolPath);
        $this->workers = (new Finder())->files()->in($workersPoolPath);
    }

    protected function canApply(): bool
    {
        return in_array(
            $this->getScenario()->getType(),
            [
                Scenario::TYPE_REBUILD,
                Scenario::TYPE_UPGRADE
            ],
            true
        );
    }

    protected function getInitMessage(): string
    {
        return 'Waiting for workers gracefully exit';
    }

    /**
     * @throws StepNotReadyException
     */
    protected function execute(): void
    {
        $timeStarted = microtime(true);

        do {
            $activeWorkers = $this->hasActiveWorkers();

            if ($activeWorkers) {
                $this->tryToRemoveOrphans();
                sleep($this->sleepForSec);
            }

            $timeElapsedInSec = microtime(true) - $timeStarted;

            if ($this->maxSleepTimeSec > 0 && $timeElapsedInSec > $this->maxSleepTimeSec) {
                throw new StepNotReadyException();
            }
        } while($activeWorkers);
    }

    protected function hasActiveWorkers(): bool
    {
        return $this->workers->count() > 0;
    }

    protected function tryToRemoveOrphans(): void
    {
        foreach ($this->workers as $worker) {
            $path = $worker;
            if ($worker instanceof SplFileInfo) {
                $path = $worker->getRealPath();
            }

            $this->removeIfOrphan($path);
        }
    }

    protected function removeIfOrphan(string $worker): void
    {
        try {
            if (
                !$this->isPidRunning($worker)
                || (string)(int)basename($worker) !== basename($worker) // If name is strictly string
            ) {
                $this->filesystem->remove($worker);
            }
        } catch (LogicException $e) {
            // Do nothing, can't decide if orphan
        }
    }

    protected function isPidRunning(string $worker): bool
    {
        $pid = (int)basename($worker);

        if (function_exists('posix_getpgid')) {
            return (bool)posix_getpgid($pid);
        }

        if ($this->filesystem->exists('/proc')) {
            return $this->filesystem->exists("/proc/$pid");
        }

        throw new LogicException('Pid status unknown');
    }
}
