<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Deployment\Step;

use App\Domain\ModuleDomain;
use App\Entity\Module;
use App\Entity\Scenario;
use App\Exception\RemoveModulesException;
use App\Marketplace\Marketplace;
use App\Repository\ModuleRepository;
use Doctrine\ORM\EntityManagerInterface;
use Exception;

final class UpdateStorage extends Step
{
    private ModuleRepository $moduleRepository;

    private EntityManagerInterface $entityManager;

    private ModuleDomain $moduleDomain;

    public function __construct(
        ModuleRepository $moduleRepository,
        EntityManagerInterface $entityManager,
        ModuleDomain $moduleDomain
    ) {
        $this->moduleRepository = $moduleRepository;
        $this->entityManager    = $entityManager;
        $this->moduleDomain     = $moduleDomain;
    }

    protected function canApply(): bool
    {
        return in_array(
            $this->getScenario()->getType(),
            [
                Scenario::TYPE_INSTALL,
                Scenario::TYPE_REBUILD,
                Scenario::TYPE_UPGRADE,
                Scenario::TYPE_UPGRADE_54_TO_55,
            ],
            true
        );
    }

    protected function getInitMessage(): string
    {
        return 'Update modules storage (change module states)';
    }

    /**
     * @throws RemoveModulesException
     * @throws Exception
     */
    protected function execute(): void
    {
        foreach ($this->getScenario()->getTransitions() as $moduleId => $transition) {
            if ($moduleId === Marketplace::CORE_MODULE_ID) {
                continue;
            }

            $module = $this->moduleRepository->findByModuleId($moduleId);
            if ($transition['stateAfter'] === Module::STATE_REMOVED) {
                if (!$module) {
                    throw RemoveModulesException::fromAlreadyRemovedModule($moduleId);
                }

                $this->entityManager->remove($module);
            } else {
                $enabledDate = ($transition['stateAfter'] === Module::STATE_ENABLED) ? time() : 0;
                $module->setState($transition['stateAfter']);
                $module->setEnabledDate($enabledDate);
            }

            if (
                $transition['stateAfter'] === Module::STATE_INSTALLED
                && $transition['stateBefore'] === Module::STATE_ENABLED
            ) {
                $module->setLastEnabledVersion(
                    $module->getVersion()
                );
            }

            if (isset($transition['versionTo'])) {
                $module->setVersion($transition['versionTo']);

                $moduleInfo = $this->moduleDomain->readModuleInfo($moduleId);
                $module->setMetaData($moduleInfo['metaData']);

                if (version_compare($transition['versionFrom'], '5.5', '<')) {
                    $module->setHasLocalFiles(true);

                    $module->setLastEnabledVersion(
                        $module->getVersion()
                    );
                }
            }

            if (
                $transition['stateBefore'] === Module::STATE_NOT_INSTALLED
                && $transition['stateAfter'] === Module::STATE_ENABLED
            ) {
                $metaData = $module->getMetaData();
                $metaData['fixtures'] = $this->moduleDomain->searchModuleInstallFixtures($moduleId);
                $module->setMetaData($metaData);
            }
        }

        $this->entityManager->flush();
    }
}
