<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Deployment\Step;

use App\Domain\XCart;
use App\Entity\Module;
use App\Entity\Scenario;
use App\Repository\ModuleRepository;
use JsonException;
use Doctrine\DBAL\Connection;
use Symfony\Component\Yaml\Yaml;

final class UpdateDatabaseSchema extends Step
{
    private XCart $XCart;

    private ModuleRepository $moduleRepository;

    private Connection $xcartConnection;

    public function __construct(
        XCart            $XCart,
        ModuleRepository $moduleRepository,
        Connection       $xcartConnection
    ) {
        $this->XCart            = $XCart;
        $this->moduleRepository = $moduleRepository;
        $this->xcartConnection  = $xcartConnection;
    }

    protected function canApply(): bool
    {
        return in_array(
            $this->getScenario()->getType(),
            [
                Scenario::TYPE_REBUILD,
                Scenario::TYPE_UPGRADE,
                Scenario::TYPE_UPGRADE_54_TO_55,
            ],
            true
        );
    }

    /**
     * @throws JsonException
     */
    protected function init(): void
    {
        if (!$this->hasStepData()) {
            $this->generateQueries();
        }
    }

    protected function getInitMessage(): string
    {
        $queries = $this->getStepData()['queries'] ?? [];

        return sprintf('Execute queries (%s queries)', count($queries));
    }

    protected function execute(): void
    {
        $this->xcartConnection->executeStatement('SET FOREIGN_KEY_CHECKS = 0;');

        $queries = $this->getStepData()['queries'] ?? [];
        foreach ($queries as $key => $query) {
            if ($query['isLoaded']) {
                continue;
            }

            $this->xcartConnection->executeStatement($query['statement']);

            $queries[$key]['isLoaded'] = true;

            $this->setStepData(['queries' => $queries]);
        }

        $this->xcartConnection->executeStatement('SET FOREIGN_KEY_CHECKS = 1;');
    }

    /**
     * @throws JsonException
     */
    private function generateQueries(): void
    {
        $data = $this->getUpdateMigrationData();

        $queries = [];

        foreach (Yaml::parse($data) as $query) {
            $queries[] = [
                'statement' => $query,
                'isLoaded'  => false,
            ];
        }

        $this->setStepData(['queries' => $queries]);
    }

    /**
     * @throws JsonException
     */
    private function getUpdateMigrationData(): string
    {
        $disabledModuleStructures = $this->getDisabledModuleStructures();

        return $this->XCart->run(
            'xcart:service:get-update-migration',
            [
                json_encode($this->getEnabledModuleTables(), JSON_THROW_ON_ERROR),
                json_encode($disabledModuleStructures['tables'], JSON_THROW_ON_ERROR),
                json_encode($disabledModuleStructures['columns'], JSON_THROW_ON_ERROR),
            ]
        );
    }

    private function getEnabledModuleTables(): array
    {
        $tables = [];

        foreach ($this->moduleRepository->getEnabledModules() as $module) {
            /** @var Module $module */
            $structure = $module->getStructure();

            if (isset($structure['tables']) && $structure['tables']) {
                $tables[] = $structure['tables'];
            }
        }

        return array_merge(...$tables);
    }

    private function getDisabledModuleStructures(): array
    {
        $modules = $this->moduleRepository->getDisabledModules();

        $tables       = [];
        $columns      = [];
        $dependencies = $this->getDependencies($modules);

        foreach ($modules as $module) {
            /** @var Module $module */
            $structure = $module->getStructure();

            if (isset($structure['tables']) && $structure['tables']) {
                $tables[] = $structure['tables'];
            }

            if (isset($structure['columns']) && $structure['columns']) {
                $columns[] = $structure['columns'];
            }

            if (isset($dependencies[$module->getModuleId()])) {
                $columns[] = $dependencies[$module->getModuleId()];
            }
        }

        $tables  = array_merge(...$tables);
        $columns = $this->getColumnsWithoutMultipleDefinitions($columns);

        return [
            'tables'  => $tables,
            'columns' => $columns,
        ];
    }

    private function getDependencies(array $modules): array
    {
        $dependencies = [];

        foreach ($modules as $module) {
            /** @var Module $module */
            $structure = $module->getStructure();

            if (isset($structure['dependencies']) && $structure['dependencies']) {
                $dependencies[] = $structure['dependencies'];
            }
        }

        return array_merge_recursive(...$dependencies);
    }

    /**
     * Return recursively merged columns without multiple column definitions
     *
     * @param array $columns
     *
     * @return array
     */
    private function getColumnsWithoutMultipleDefinitions(array $columns): array
    {
        $columns = array_merge_recursive(...$columns);

        // exclude multiple column definitions after array_merge_recursive
        foreach ($columns as $table => $columnsList) {
            foreach ($columnsList as $columnName => $columnDefinition) {
                if (is_array($columnDefinition)) {
                    $columns[$table][$columnName] = end($columnDefinition);
                }
            }

        }

        return $columns;
    }
}
