<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Deployment\Step;

use App\Domain\ModuleDomain;
use App\Entity\Scenario;
use App\Repository\ModuleRepository;

final class SyncFilesystemModulesWithDatabase extends Step
{
    private ModuleDomain $moduleDomain;

    private ModuleRepository $moduleRepository;

    public function __construct(
        ModuleDomain     $moduleDomain,
        ModuleRepository $moduleRepository
    ) {
        $this->moduleDomain     = $moduleDomain;
        $this->moduleRepository = $moduleRepository;
    }

    protected function canApply(): bool
    {
        return $this->getScenario()->getType() === Scenario::TYPE_REBUILD;
    }

    protected function getInitMessage(): string
    {
        return 'Sync filesystem modules with database';
    }

    protected function execute(): void
    {
        $filesystemModules    = [];
        $databaseModules = $this->moduleRepository->getAllIds();

        foreach ($this->moduleDomain->searchLocalModules() as $moduleId) {
            $filesystemModules[] = $moduleId;
        }

        $this->removeNonexistentModulesFromDatabase($filesystemModules, $databaseModules);
        $this->addNewModulesToDatabase($filesystemModules, $databaseModules);
    }

    private function removeNonexistentModulesFromDatabase(
        array $filesystemModules,
        array $databaseModules
    ): void {
        $nonexistentModules = array_diff($databaseModules, $filesystemModules);

        $this->moduleRepository->removeModules($nonexistentModules);
    }

    private function addNewModulesToDatabase(
        array $filesystemModules,
        array $databaseModules
    ): void {
        $newModules = array_map(
            fn ($moduleId) => $this->moduleDomain->readModuleInfo($moduleId),
            array_diff($filesystemModules, $databaseModules)
        );

        $this->moduleRepository->createModules($newModules);
    }
}
