<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Deployment\Step;

use App\Entity\Scenario;
use App\Output\XCartOutputInterface;
use Symfony\Component\Workflow\Event\TransitionEvent;

abstract class Step
{
    protected ?TransitionEvent $event;

    abstract protected function canApply(): bool;

    abstract protected function getInitMessage(): string;

    abstract protected function execute(): void;

    public function onRebuildStep(TransitionEvent $event): void
    {
        $this->event = $event;

        $this->run();
    }
    protected function run(): void
    {
        $this->init();

        if ($this->canApply()) {
            $output = $this->getOutput();

            $output->addInitMessage($this->getInitMessage());

            $this->execute();

            $this->end();

            $output->addEndMessage($this->getEndMessage());
        }
    }

    protected function init(): void
    {
    }

    protected function end(): void
    {
    }

    protected function getEndMessage(): string
    {
        return '[OK]';
    }

    protected function getOutput(): XCartOutputInterface
    {
        return $this->event->getContext()['output'];
    }

    protected function getScenario(): Scenario
    {
        /** @var Scenario $scenario */
        $scenario = $this->event->getSubject();

        return $scenario;
    }

    protected function hasStepData(string $step = ''): bool
    {
        return $this->getScenario()->hasStepData($step ?: static::class);
    }

    protected function getStepData(string $step = ''): array
    {
        return $this->getScenario()->getStepData($step ?: static::class);
    }

    protected function setStepData(array $data, string $step = ''): void
    {
        $this->getScenario()->setStepData($step ?: static::class, $data);
    }
}
