<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Deployment\Step;

use App\Domain\XCart;
use App\Entity\Scenario;
use App\Operation\Build\Configuration\GenerateAutoloader;
use App\Operation\Build\Configuration\GenerateBundles;
use App\Operation\Build\Configuration\GenerateDoctrineMappings;
use App\Operation\Build\Configuration\GenerateModules;
use App\Operation\Build\Configuration\GeneratePHPUnitConfiguration;
use App\Operation\Build\Configuration\GenerateSkinModel;
use Symfony\Component\Yaml\Yaml;

final class SetConfiguration extends Step
{
    private XCart $XCart;

    private GenerateAutoloader $generateAutoloader;

    private GeneratePHPUnitConfiguration $generatePHPUnitConfiguration;

    private GenerateModules $generateModules;

    private GenerateDoctrineMappings $generateDoctrineMappings;

    private GenerateSkinModel $generateSkinModel;

    private GenerateBundles $generateBundles;

    public function __construct(
        XCart $XCart,
        GenerateAutoloader $generateAutoloader,
        GeneratePHPUnitConfiguration $generatePHPUnitConfiguration,
        GenerateModules $generateModules,
        GenerateDoctrineMappings $generateDoctrineMappings,
        GenerateSkinModel $generateSkinModel,
        GenerateBundles $generateBundles
    ) {
        $this->XCart                        = $XCart;
        $this->generateAutoloader           = $generateAutoloader;
        $this->generatePHPUnitConfiguration = $generatePHPUnitConfiguration;
        $this->generateModules              = $generateModules;
        $this->generateDoctrineMappings     = $generateDoctrineMappings;
        $this->generateSkinModel            = $generateSkinModel;
        $this->generateBundles              = $generateBundles;
    }

    protected function canApply(): bool
    {
        return in_array(
            $this->getScenario()->getType(),
            [
                Scenario::TYPE_INSTALL,
                Scenario::TYPE_REBUILD,
                Scenario::TYPE_UPGRADE,
                Scenario::TYPE_UPGRADE_54_TO_55,
            ],
            true
        );
    }

    protected function getInitMessage(): string
    {
        return 'Set XC configuration';
    }

    protected function execute(): void
    {
        $this->XCart->writeToFile('vendor/autoload_xcart.php', ($this->generateAutoloader)());
        $this->XCart->writeToFile('phpunit.xml', ($this->generatePHPUnitConfiguration)());
        $this->XCart->writeToFile('config/dynamic/xcart_modules.yaml', Yaml::dump(($this->generateModules)(), 5));
        $this->XCart->writeToFile('config/dynamic/xcart_doctrine_mappings.yaml', Yaml::dump(($this->generateDoctrineMappings)(), 5));
        $this->XCart->writeToFile('config/dynamic/xcart_skin_model.yaml', Yaml::dump(($this->generateSkinModel)(), 5));

        $bundles =  var_export(($this->generateBundles)(), true);
        $this->XCart->writeToFile('config/dynamic/xcart_bundles.php', "<?php return {$bundles};");
    }
}
