<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Deployment\Step;

use App\Domain\XCart;
use App\Entity\Module;
use App\Entity\Scenario;
use Symfony\Component\Filesystem\Filesystem;

final class RemoveModules extends Step
{
    private XCart $XCart;

    private Filesystem $filesystem;

    public function __construct(
        XCart $XCart,
        Filesystem $filesystem
    ) {
        $this->XCart      = $XCart;
        $this->filesystem = $filesystem;
    }

    protected function init(): void
    {
        $this->generateRemoveIds();
    }

    private function generateRemoveIds(): void
    {
        $this->setStepData(['removeIds' => $this->getModulesToRemove()]);
    }

    protected function canApply(): bool
    {
        return $this->getScenario()->getType() === Scenario::TYPE_REBUILD
            && $this->getStepData()['removeIds'];
    }

    protected function getInitMessage(): string
    {
        return 'Remove modules';
    }

    protected function execute(): void
    {
        $modulesToRemove = $this->getStepData()['removeIds'] ?? [];
        foreach ($modulesToRemove as $moduleId) {
            $path = $this->XCart->getModulesPath() . str_replace('-', '/', $moduleId);

            if ($this->filesystem->exists($path)) {
                $this->filesystem->remove($path);
            }
        }
    }

    private function getModulesToRemove(): ?array
    {
        $moduleIds = [];
        foreach ($this->getScenario()->getTransitions() as $moduleId => $transition) {
            if ($transition['stateAfter'] === Module::STATE_REMOVED) {
                $moduleIds[] = $moduleId;
            }
        }

        return $moduleIds;
    }
}
