<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Deployment\Step;

use App\Domain\Backup;
use App\Domain\ServiceToolConsole;
use App\Domain\XCart;
use App\Entity\Scenario;
use App\Marketplace\Marketplace;
use App\Operation\ExecuteShellCommand;
use CallbackFilterIterator;
use FilesystemIterator;
use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;
use SplFileInfo;
use Symfony\Component\Filesystem\Filesystem;

final class MovePacks extends Step
{
    private string $sourcePath;

    private string $packsPath;

    private Filesystem $filesystem;

    private Backup $backup;

    private ServiceToolConsole $serviceToolConsole;

    private ExecuteShellCommand $executeShellCommand;

    public function __construct(
        XCart $XCart,
        Filesystem $filesystem,
        Backup $backup,
        ServiceToolConsole $serviceToolConsole,
        ExecuteShellCommand $executeShellCommand
    ) {
        $this->sourcePath          = $XCart->getSourcePath();
        $this->packsPath           = $XCart->getPacksPath();
        $this->filesystem          = $filesystem;
        $this->backup              = $backup;
        $this->serviceToolConsole  = $serviceToolConsole;
        $this->executeShellCommand = $executeShellCommand;
    }

    protected function canApply(): bool
    {
        $scenario         = $this->getScenario();
        $scenarioType     = $scenario->getType();
        $scenarioMetadata = $scenario->getMetaData();

        return (
            ($scenarioType === Scenario::TYPE_REBUILD && isset($scenarioMetadata['packPaths']))
            || ($scenarioType === Scenario::TYPE_UPGRADE && isset($scenarioMetadata['upgradeEntries']))
        );
    }

    protected function getInitMessage(): string
    {
        return 'Move packs';
    }

    protected function execute(): void
    {
        switch ($this->getScenario()->getType()) {
            case Scenario::TYPE_REBUILD:
                $this->moveInstallPacks();
                break;
            case Scenario::TYPE_UPGRADE:
                $this->moveUpgradePacks();
                break;
        }
    }

    private function moveInstallPacks(): void
    {
        $packsToMove = $this->getScenario()->getMetaData()['packPaths'] ?? [];

        foreach ($packsToMove as $packPath) {
            $this->filesystem->remove([
                "{$packPath}/.hash",
                "{$packPath}/.phar",
            ]);

            $iterator = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($packPath, FilesystemIterator::SKIP_DOTS), RecursiveIteratorIterator::SELF_FIRST);

            $iterator = new CallbackFilterIterator($iterator, function (SplFileInfo $current) {
                if (str_starts_with($current->getFilename(), '._')) {
                    return false;
                }

                return true;
            });


            $this->filesystem->mirror($packPath, $this->sourcePath, $iterator);
            $this->filesystem->remove($packPath);
        }
    }

    private function moveUpgradePacks(): void
    {
        $this->backup->create(
            (string) $this->getScenario()->getId()
        );

        $upgradeEntries = $this->getScenario()->getMetaData()['upgradeEntries'] ?? [];
        $transitions    = $this->getScenario()->getTransitions();

        foreach ($upgradeEntries as $moduleId => $moduleUpgradeEntries) {
            $moduleVersion = $transitions[$moduleId]['versionTo'];
            $packPath      = "{$this->packsPath}{$moduleId}-v{$moduleVersion}";

            foreach ($moduleUpgradeEntries as $key => $upgradeEntry) {
                $filepath = $upgradeEntry['filepath'];

                $originFilePath = "{$packPath}/{$filepath}";
                $targetFilePath = "{$this->sourcePath}{$filepath}";

                switch ($upgradeEntry['type']) {
                    case 'modified': // the file is changed in the new version of the module
                        $this->backup->addReplaceRecord($filepath);
                        $this->filesystem->copy($originFilePath, $targetFilePath, true);
                        break;
                    case 'removed': // the file does not exist in the old version of the module
                        $this->backup->addCreateRecord($filepath);
                        $this->filesystem->copy($originFilePath, $targetFilePath);
                        break;
                    case 'added': // the file does not exist in the new version of the module
                        $this->backup->addReplaceRecord($filepath);
                        $this->filesystem->remove($targetFilePath);
                        break;
                }

                unset($moduleUpgradeEntries[$key]);
                $metaData = $this->getScenario()->getMetaData();
                $metaData['upgradeEntries'][$moduleId] = $moduleUpgradeEntries;
                $this->getScenario()->setMetaData($metaData);
            }

            $this->filesystem->remove($packPath);
        }

        if (isset($upgradeEntries[Marketplace::CORE_MODULE_ID])) {
            $this->clearServiceToolCache();
        }
    }

    private function clearServiceToolCache(): void
    {
        ($this->executeShellCommand)(['rm', '-rf', 'service-tool/var/cache'], $this->sourcePath);
        opcache_reset();
        $this->serviceToolConsole->run('cache:clear');
    }
}
