<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Deployment\Step;

use App\Domain\ModuleDomain;
use App\Domain\XCart;
use App\Entity\Scenario;
use App\Repository\ModuleRepository;
use Exception;

final class LoadFixtures extends Step
{
    private ModuleRepository $moduleRepository;

    private XCart $XCart;

    private ModuleDomain $moduleDomain;

    public function __construct(
        XCart $XCart,
        ModuleDomain $moduleDomain,
        ModuleRepository $moduleRepository
    ) {
        $this->XCart            = $XCart;
        $this->moduleDomain     = $moduleDomain;
        $this->moduleRepository = $moduleRepository;
    }

    protected function canApply(): bool
    {
        return in_array(
            $this->getScenario()->getType(),
            [
                Scenario::TYPE_INSTALL,
                Scenario::TYPE_REBUILD,
            ],
            true
        );
    }

    protected function init(): void
    {
        if (!$this->hasStepData()) {
            $this->generateFixtures();
        }
    }

    protected function getInitMessage(): string
    {
        $fixtures = $this->getStepData()['fixtures'] ?? [];

        return sprintf('Load fixtures (%s files)', count($fixtures));
    }

    /**
     * @throws Exception
     */
    protected function execute(): void
    {
        $fixtures = $this->getStepData()['fixtures'] ?? [];

        if ($fixturesToLoad = $this->getFixturesToLoad($fixtures)) {
            $output = $this->XCart->run('xcart:service:load-fixtures', [implode(',', $fixturesToLoad)]);

            $output = array_filter(preg_split('/\r\n|\n|\r/', $output));
            if (
                !empty($output)
                && $result = array_shift($output)
            ) {
                if (substr($result, 0, 5) === 'ERROR') {
                    $unloadedFixtures = [];
                    $unloadedModuleFixtures = [];
                    foreach ($fixtures as $key => $fixture) {
                        if ($fixture['isLoaded']) {
                            continue;
                        }

                        if (in_array($fixture['path'], $output, true)) {
                            $fixtures[$key]['isLoaded'] = true;
                            continue;
                        }

                        $unloadedFixtures[] = $fixture['path'];
                        if (!empty($fixture['moduleId'])) {
                            $unloadedModuleFixtures[$fixture['moduleId']][] = $fixture['path'];
                        }
                    }

                    foreach ($unloadedModuleFixtures as $moduleId => $paths) {
                        $module = $this->moduleRepository->findByModuleId($moduleId);

                        if ($module) {
                            $metaData = $module->getMetaData();
                            $metaData['fixtures'] = $fixtures;
                            $module->setMetaData($metaData);
                        }
                    }

                    $this->setStepData(['fixtures' => $fixtures]);
                    $result .= PHP_EOL . 'Unloaded fixtures: '. PHP_EOL . implode(PHP_EOL, $unloadedFixtures);

                    throw new \RuntimeException(substr($result, 6));
                }
            }

            $this->XCart->run('xcart:service:post-fixtures-load', [$this->getScenario()->getType()]);
        }

        $this->moduleRepository->removeAllFixtures();
    }

    private function generateFixtures(): void
    {
        $fixtures   = [];
        $scenario = $this->getScenario();

        $metaData   = $scenario->getMetaData();
        $sourcePath = $this->XCart->getSourcePath();

        foreach ($metaData['data'] ?? [] as $file) {
            $fixtures[] = [
                'path'     => "{$sourcePath}{$file}",
                'isLoaded' => false,
            ];
        }

        foreach ($this->moduleRepository->getModulesWithNotEmptyFixtures() as $module) {
            foreach ($module->getMetaData()['fixtures'] ?? [] as $fixture) {
                $fixtures[] = [
                    'path'     => (string) $fixture,
                    'isLoaded' => false,
                    'moduleId' => $module->getModuleId()
                ];
            }
        }

        foreach ($metaData['demoData'] ?? [] as $file) {
            $fixtures[] = [
                'path'     => "{$sourcePath}{$file}",
                'isLoaded' => false,
            ];
        }

        $this->setStepData(['fixtures' => $fixtures]);
    }

    private function getFixturesToLoad(array $fixtures): array
    {
        return array_values(
            array_map(
                static fn ($f): string => $f['path'],
                array_filter(
                    $fixtures,
                    static fn ($f): bool => !$f['isLoaded']
                )
            )
        );
    }
}
