<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Deployment\Step;

use App\Domain\XCart;
use App\Entity\Scenario;
use Doctrine\DBAL\Connection;
use Symfony\Component\Yaml\Yaml;

final class InitDatabase extends Step
{
    private XCart $XCart;

    private Connection $connection;

    public function __construct(
        XCart            $XCart,
        Connection       $connection
    ) {
        $this->XCart            = $XCart;
        $this->connection  = $connection;
    }

    protected function canApply(): bool
    {
        return $this->getScenario()->getType() === Scenario::TYPE_INSTALL
            && ($this->getStepData()['queries'] ?? []);
    }

    protected function init(): void
    {
        if (!$this->hasStepData()) {
            $this->generateQueries();
        }
    }

    protected function getInitMessage(): string
    {
        $queries = $this->getStepData()['queries'] ?? [];

        return sprintf('Execute queries (%s queries)', count($queries));
    }

    protected function execute(): void
    {
        $this->connection->executeStatement('SET FOREIGN_KEY_CHECKS = 0;');

        $queries = $this->getStepData()['queries'] ?? [];
        foreach ($queries as $key => $query) {
            if ($query['isLoaded']) {
                continue;
            }

            $this->connection->executeStatement($query['statement']);

            $queries[$key]['isLoaded'] = true;

            $this->setStepData(['queries' => $queries]);
        }

        $this->connection->executeStatement('SET FOREIGN_KEY_CHECKS = 1;');
    }

    private function generateQueries(): void
    {
        $data = $this->XCart->run('xcart:service:get-create-migration');

        $queries = [];

        foreach (Yaml::parse($data) as $query) {
            $queries[] = [
                'statement' => $query,
                'isLoaded'  => false,
            ];
        }

        $this->setStepData(['queries' => $queries]);
    }
}
