<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Deployment\Step;

use App\Domain\XCart;
use App\Entity\Scenario;
use App\Operation\ExecuteShellCommand;

final class ClearXCartCache extends Step
{
    private const MAX_ATTEMPTS_TO_REMOVE_CACHE = 3;

    private XCart $XCart;

    private ExecuteShellCommand $executeShellCommand;

    public function __construct(
        XCart $XCart,
        ExecuteShellCommand $executeShellCommand
    ) {
        $this->XCart = $XCart;
        $this->executeShellCommand = $executeShellCommand;
    }

    protected function canApply(): bool
    {
        return in_array(
            $this->getScenario()->getType(),
            [
                Scenario::TYPE_INSTALL,
                Scenario::TYPE_REBUILD,
                Scenario::TYPE_UPGRADE,
                Scenario::TYPE_UPGRADE_54_TO_55,
            ],
            true
        );
    }

    protected function getInitMessage(): string
    {
        return 'Clear xcart cache';
    }

    protected function execute(): void
    {
        $this->removeCache();

        $this->XCart->run('cache:clear', ['--no-optional-warmers']);
        $this->XCart->run('cache:pool:clear', ['cache.global_clearer']);
        $this->XCart->run('cache:warmup');

        if ($this->XCart->run('xcart:ll:check-jwt-keys') === 'fail') {
            $this->XCart->run('lexik:jwt:generate-keypair', ['--overwrite']);
        }
    }

    private function removeCache(): void
    {
        $attempt = 0;

        while (file_exists($this->XCart->getSourcePath() . 'var/cache') &&  $attempt < self::MAX_ATTEMPTS_TO_REMOVE_CACHE) {
            ($this->executeShellCommand)(['mv', 'var/cache', 'var/cache-tmp'], $this->XCart->getSourcePath());
            ($this->executeShellCommand)(['rm', '-rf', 'var/cache-tmp'], $this->XCart->getSourcePath());

            $attempt++;
        }
    }
}
