<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Deployment\Step;

use App\Domain\HookManagerDomain;
use App\Domain\XCart;
use App\Entity\Scenario;
use App\Exception\HookManagerException;
use App\Marketplace\Marketplace;

final class CallServiceHooks extends CallHooks
{
    private HookManagerDomain $hookManagerDomain;

    public function __construct(
        XCart $XCart,
        HookManagerDomain $hookManagerDomain
    ) {
        parent::__construct($XCart);

        $this->hookManagerDomain = $hookManagerDomain;
    }

    protected function canApply(): bool
    {
        return (
            in_array($this->getScenario()->getType(), [Scenario::TYPE_UPGRADE, Scenario::TYPE_UPGRADE_54_TO_55], true)
            && ($this->getScenario()->getTransitions()[Marketplace::CORE_MODULE_ID] ?? false)
        );
    }

    protected function getInitMessage(): string
    {
        return 'Run service hooks';
    }

    protected function generateHookList(): void
    {
        $hooks = [
            'upgrade' => [],
        ];

        $transitions = $this->getScenario()->getTransitions();

        if ($transition = $transitions[Marketplace::CORE_MODULE_ID] ?? null) {
            $versionFrom = $transition['versionFrom'] ?? '';
            $versionTo   = $transition['versionTo'] ?? '';

            if ($versionFrom !== $versionTo) {
                $hooks['upgrade'][] = [
                    'versionFrom' => $versionFrom,
                    'versionTo'   => $versionTo,
                ];
            }
        }

        $this->setStepData(['hooks' => $hooks]);
    }

    /**
     * @throws HookManagerException
     */
    protected function execute(): void
    {
        foreach ($this->getStepData()['hooks'] as $type => $hooks) {
            foreach ($hooks as $hook) {
                $this->hookManagerDomain->runHook([
                    'hookType'    => $type,
                    'versionFrom' => $hook['versionFrom'] ?? '',
                    'versionTo'   => $hook['versionTo'] ?? '',
                ]);
            }
        }
    }
}
