<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Deployment\Step;

use App\Domain\ModuleDomain;
use App\Domain\XCart;
use App\Entity\Module;
use App\Entity\Scenario;
use App\Marketplace\Marketplace;
use App\Repository\ModuleRepository;

final class CallPostHooks extends CallHooks
{
    private ModuleRepository $moduleRepository;

    private ModuleDomain $moduleDomain;

    public function __construct(
        XCart            $XCart,
        ModuleDomain     $moduleDomain,
        ModuleRepository $moduleRepository
    ) {
        parent::__construct($XCart);

        $this->moduleDomain     = $moduleDomain;
        $this->moduleRepository = $moduleRepository;
    }

    protected function getInitMessage(): string
    {
        return 'Run install, enable, rebuild and upgrade hooks';
    }

    protected function generateHookList(): void
    {
        // hook order matters
        $hooks = [
            'install' => [],
            'enable'  => [],
            'upgrade' => [],
            'rebuild' => [],
        ];

        foreach ($this->getScenario()->getTransitions() as $moduleId => $transition) {
            if (
                $transition['stateBefore'] === Module::STATE_NOT_INSTALLED
                && in_array(
                    $transition['stateAfter'],
                    [Module::STATE_INSTALLED, Module::STATE_ENABLED],
                    true
                )
            ) {
                $hooks['install'][] = ['moduleId' => $moduleId];
            }

            if ($this->getScenario()->getType() === Scenario::TYPE_INSTALL) {
                $hooks['install'][] = ['moduleId' => Marketplace::CORE_MODULE_ID];
            }

            if (
                $transition['stateAfter'] === Module::STATE_ENABLED
                && $transition['stateBefore'] !== Module::STATE_ENABLED
            ) {
                $hooks['enable'][] = ['moduleId' => $moduleId];
            }

            if ($transition['stateAfter'] === Module::STATE_ENABLED) {
                $versionFrom = $transition['versionFrom'] ?? '';
                $versionTo   = $transition['versionTo'] ?? '';

                if ($versionFrom && $versionTo) {
                    $hooks['upgrade'][] = [
                        'moduleId'    => $moduleId,
                        'versionFrom' => $versionFrom,
                        'versionTo'   => $versionTo,
                    ];
                }
            }
        }

        foreach ($this->moduleRepository->getEnabledIds() as $moduleId) {
            $hooks['rebuild'][] = ['moduleId' => $moduleId];
        }

        $this->setStepData(['hooks' => $hooks]);
    }

    protected function execute(): void
    {
        $this->loadLabels();

        parent::execute();
    }

    private function loadLabels(): void
    {
        $yamlFiles = [];

        foreach ($this->getScenario()->getTransitions() as $moduleId => $transition) {
            if (
                (
                    $transition['transitionType'] === 'upgrade'
                    && $transition['stateAfter'] === Module::STATE_ENABLED
                )
                || (
                    $transition['transitionType'] === 'enable'
                    && isset($transition['versionFrom'], $transition['versionTo'])
                    && $transition['versionFrom'] !== $transition['versionTo']
                )
            ) {
                $files = $this->moduleDomain->searchModuleInstallFixtures($moduleId);

                if ($files) {
                    $yamlFiles[] = $files;
                }
            }
        }

        if ($yamlFiles) {
            $this->XCart->run(
                'xcart:service:load-labels',
                array_merge(...$yamlFiles)
            );
        }
    }
}
