<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Deployment;

use App\Entity\Scenario;
use App\Event\RunnerEvent;
use App\Exception\ClearServiceToolCacheException;
use App\Exception\StepNotReadyException;
use App\Output\XCartOutputInterface;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;
use Throwable;

final class Runner
{
    private EventDispatcherInterface $eventDispatcher;

    private bool $isInterrupted;

    public function __construct(EventDispatcherInterface $eventDispatcher)
    {
        $this->eventDispatcher = $eventDispatcher;
        $this->isInterrupted   = false;
    }

    /**
     * @noinspection PhpRedundantCatchClauseInspection
     * @throws ClearServiceToolCacheException
     */
    public function __invoke(Scenario $scenario, XCartOutputInterface $output)
    {
        pcntl_signal(SIGTERM, [$this, 'stop']);
        pcntl_signal(SIGINT, [$this, 'stop']);

        $runnerEvent = new RunnerEvent();
        $runnerEvent
            ->setScenario($scenario)
            ->setOutput($output);

        $this->eventDispatcher->dispatch($runnerEvent, 'service-tool.runner.start');

        while (!$scenario->isDone()) {
            pcntl_signal_dispatch();

            if ($this->isInterrupted) {
                return;
            }

            try {
                $this->eventDispatcher->dispatch($runnerEvent, 'service-tool.runner.run-next-step');
            } catch (ClearServiceToolCacheException $e) {
                // After move_packs step a new runtime should start with a new up-to-date cache
                throw $e;
            } catch (StepNotReadyException $e) {
                $output
                    ->addErrorMessage('[KO]')
                    ->addErrorMessage($e->getMessage());
                // Continue, state not changed
            } catch (Throwable $e) {
                $runnerEvent->setError($e->getMessage());
                $this->eventDispatcher->dispatch($runnerEvent, 'service-tool.runner.abort');

                $output
                    ->addErrorMessage('[KO]')
                    ->addErrorMessage($e->getMessage());

                return;
            }
        }

        $this->eventDispatcher->dispatch($runnerEvent, 'service-tool.runner.finish');
    }

    public function stop()
    {
        $this->isInterrupted = true;
    }
}
