<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace App\DataTransformer;

use ApiPlatform\Core\DataTransformer\DataTransformerInterface;
use App\DTO\Input\ScenarioInput;
use App\DTO\Input\UpgradeScenarioInput;
use App\Entity\Scenario;
use App\Exception\RebuildIsAlreadyStartedException;
use App\Operation\Build\Upgrade\GenerateTransitions;
use App\Operation\Build\RebuildFlag;
use Symfony\Component\HttpFoundation\JsonResponse;

class UpgradeScenarioInputDataTransformer implements DataTransformerInterface
{
    private GenerateTransitions $generateTransitions;

    private RebuildFlag $rebuildFlag;

    public function __construct(
        GenerateTransitions $generateTransitions,
        RebuildFlag $rebuildFlag
    ) {
        $this->generateTransitions = $generateTransitions;
        $this->rebuildFlag         = $rebuildFlag;
    }

    /**
     * @param ScenarioInput $object
     *
     * @return Scenario|JsonResponse
     */
    public function transform($object, string $to, array $context = [])
    {
        try {
            $this->rebuildFlag->check();
        } catch (RebuildIsAlreadyStartedException $e) {
            return new JsonResponse([
                'error'      => $e->getMessage(),
                'errorType'  => 'alreadyStarted',
                'scenarioId' => $e->getScenarioId(),
            ]);
        }

        $modulesToUpgrade = [];
        foreach ($object->modulesToUpgrade as $moduleToUpgrade) {
            $modulesToUpgrade[$moduleToUpgrade['id']] = $moduleToUpgrade;
        }

        $transitions = ($this->generateTransitions)($modulesToUpgrade);

        $scenario = new Scenario();
        $scenario->setType(Scenario::TYPE_UPGRADE);
        $scenario->setTransitions($transitions);
        $scenario->setMetaData(['modulesToUpgrade' => $modulesToUpgrade]);

        return $scenario;
    }

    public function supportsTransformation($data, string $to, array $context = []): bool
    {
        if ($data instanceof Scenario) {
            return false;
        }

        return (
            $to === Scenario::class
            && $context['input']['class'] === UpgradeScenarioInput::class
        );
    }
}