<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace App\DataTransformer;

use ApiPlatform\Core\DataTransformer\DataTransformerInterface;
use App\DTO\Input\ScenarioInput;
use App\Entity\Scenario;
use App\Event\RunnerEvent;
use App\Exception\BuildException;
use App\Exception\RebuildIsAlreadyStartedException;
use App\Operation\Build\GenerateTransitions;
use App\Operation\Build\RebuildFlag;
use App\Operation\Build\RemoveActiveScenario;
use App\Repository\ModuleRepository;
use JsonException;
use MJS\TopSort\CircularDependencyException;
use MJS\TopSort\ElementNotFoundException;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

class ScenarioInputDataTransformer implements DataTransformerInterface
{
    private const SKIN_STANDARD = 'standard';

    private GenerateTransitions $generateTransitions;

    private RemoveActiveScenario $removeActiveScenario;

    private RebuildFlag $rebuildFlag;

    private EventDispatcherInterface $eventDispatcher;

    private ModuleRepository $moduleRepository;

    public function __construct(
        GenerateTransitions $generateTransitions,
        RemoveActiveScenario $removeActiveScenario,
        RebuildFlag $rebuildFlag,
        EventDispatcherInterface $eventDispatcher,
        ModuleRepository $moduleRepository
    ) {
        $this->generateTransitions  = $generateTransitions;
        $this->removeActiveScenario = $removeActiveScenario;
        $this->rebuildFlag          = $rebuildFlag;
        $this->eventDispatcher      = $eventDispatcher;
        $this->moduleRepository     = $moduleRepository;
    }

    /**
     * @param ScenarioInput $object
     *
     * @return Scenario|JsonResponse
     * @throws BuildException
     * @throws JsonException
     * @throws CircularDependencyException
     * @throws ElementNotFoundException
     */
    public function transform($object, string $to, array $context = [])
    {
        $force = $object->force ?? false;

        if ($force) {
            ($this->removeActiveScenario)();
        } else {
            try {
                $this->rebuildFlag->check();
            } catch (RebuildIsAlreadyStartedException $e) {
                return new JsonResponse([
                    'error'      => $e->getMessage(),
                    'errorType'  => 'alreadyStarted',
                    'scenarioId' => $e->getScenarioId(),
                ]);
            }
        }

        $modulesToEnable  = $object->modulesToEnable ?? [];
        $modulesToDisable = $object->modulesToDisable ?? [];
        $modulesToRemove  = $object->modulesToRemove ?? [];

        $skinToSwitch = $object->skinToSwitch ?? '';

        if ($skinToSwitch) {
            if ($skinToSwitch === self::SKIN_STANDARD) {
                $modulesToDisable += $this->moduleRepository->getSortedEnabledSkinModuleIds();
            } else {
                $modulesToEnable[] = $skinToSwitch;
            }
        }

        $transitions = ($this->generateTransitions)($modulesToEnable, $modulesToDisable, $modulesToRemove);

        $scenario = new Scenario();
        $scenario->setType(Scenario::TYPE_REBUILD);
        $scenario->setTransitions($transitions);

        $runnerEvent = new RunnerEvent();
        $runnerEvent->setScenario($scenario);

        $this->eventDispatcher->dispatch($runnerEvent, 'service-tool.runner.start');

        return $scenario;
    }

    public function supportsTransformation($data, string $to, array $context = []): bool
    {
        if ($data instanceof Scenario) {
            return false;
        }

        return (
            $to === Scenario::class
            && $context['input']['class'] === ScenarioInput::class
        );
    }
}
