<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace App\DataTransformer;

use ApiPlatform\Core\DataTransformer\DataTransformerInterface;
use App\DTO\Output\ModuleOutput;
use App\Entity\CheckRequirementsResult;
use App\Entity\Module;
use App\Exception\CircularDependencyException;
use App\Logic\CheckRequirements;
use App\Marketplace\Marketplace;
use App\Repository\ModuleRepository;

class ModuleOutputDataTransformer implements DataTransformerInterface
{
    private Marketplace $marketplace;

    private CheckRequirements $checkRequirements;

    private ModuleRepository $moduleRepository;

    private array $config;

    public function __construct(
        $config,
        Marketplace $marketplace,
        CheckRequirements $checkRequirements,
        ModuleRepository $moduleRepository
    ) {
        $this->marketplace       = $marketplace;
        $this->checkRequirements = $checkRequirements;
        $this->moduleRepository  = $moduleRepository;
        $this->config            = $config;
    }

    /**
     * @param Module $object
     *
     * @throws CircularDependencyException
     * @throws \JsonException
     */
    public function transform($object, string $to, array $context = []): ModuleOutput
    {
        $output            = new ModuleOutput();
        $output->id        = $object->getModuleId();
        $output->state     = $object->getState();
        $output->version   = $object->getVersion();
        $output->author    = $object->getAuthor();
        $output->name      = $object->getName();

        $metadata    = $object->getMetaData();
        $dynamicData = $this->marketplace->getModule($object->getAuthor(), $object->getName());

        $output->type          = $metadata['type'];
        $output->authorName    = $metadata['authorName'];
        $output->moduleName    = $metadata['moduleName'];
        $output->description   = $metadata['description'];
        $output->enabledDate   = $object->getEnabledDate();
        $output->isCustom      = empty($dynamicData);
        $output->hasLocalFiles = $object->hasLocalFiles();
        $output->pageUrl       = $dynamicData['pageURL'] ?? '';

        if ($output->pageUrl) {
            $appStoreUrl = $this->config['appstore_url'] ?? '';

            if ($appStoreUrl) {
                $pageUrlParts = parse_url($output->pageUrl);
                $appStoreHost = parse_url($appStoreUrl, PHP_URL_HOST);

                if ($appStoreHost && ($pageUrlParts['host'] ?? '') === $appStoreHost) {
                    $pageUrlQuery = $pageUrlParts['query'] ?? '';
                    if (
                        !$pageUrlQuery
                        || (
                            substr($pageUrlQuery, 0, strlen('xc5_shop_identifier=')) !== 'xc5_shop_identifier='
                            && strpos($pageUrlQuery, '&xc5_shop_identifier=') === false
                        )
                    ) {
                        $output->pageUrl .= ((strpos($output->pageUrl, '?') === false) ? '?' : '&')
                            . 'xc5_shop_identifier=' . $this->marketplace->retrieveStoreIdentity();
                    }
                }
            }
        }

        $transitions = $context['transitions']
            ? json_decode($context['transitions'], true, 512, JSON_THROW_ON_ERROR)
            : [];

        $output->scenarioState = $this->getScenarioState($object, $transitions);

        [$actions, $warning] = $this->getModuleActionsAndWarning(
            $object,
            $transitions,
            $output->scenarioState
        );

        $output->actions = $actions;
        $output->warning = $warning;

        return $output;
    }

    public function supportsTransformation($data, string $to, array $context = []): bool
    {
        return $to === ModuleOutput::class && $data instanceof Module;
    }

    /**
     * @throws \JsonException
     * @throws CircularDependencyException
     */
    private function getModuleActionsAndWarning(
        Module $module,
        array $transitions,
        string $scenarioModuleState
    ): array {
        if (version_compare($module->getVersion(), '5.5', '<')) {
            return [
                ['remove' => true],
                [],
            ];
        }

        $isSkin = $module->isSkin();

        $actions = [
            'manageLayout'     => $isSkin,
            'showSettingsForm' => $module->getMetaData()['showSettingsForm'],
            'remove'           => true,
        ];

        $warning = [];

        if ($isSkin) {
            $actions['switchState'] = false;
        } else {
            $this->simulateSwitchAction($module, $transitions, $scenarioModuleState);
            $requirements = $this->checkRequirements->checkActions();

            $actions['remove']      = !$requirements->getCantDisable();
            $actions['switchState'] = !$requirements->getFailedActions();

            $disallowedActions = $scenarioModuleState === Module::STATE_ENABLED
                ? $requirements->getCantDisable()
                : $requirements->getCantEnable();

            if ($disallowedActions) {
                // info is needed only about the main disallowed action
                [$moduleId, $reason, $data] = end($disallowedActions);

                if ($reason === CheckRequirementsResult::CANT_DISABLE_BY_MODULE_PROPERTY_REASON) {
                    $actions['remove'] = true;
                }

                $warning = [
                    'message' => $reason,
                    'params'  => $this->prepareModulesList($data),
                ];

            }
        }

        return [$actions, $warning];
    }

    private function simulateSwitchAction(
        Module $module,
        array $transitions,
        string $scenarioModuleState
    ): void {
        if (!$this->checkRequirements->isSetDependencyMap()) {
            $dependencyMap = $this->moduleRepository->getDependencyMap();
            $this->checkRequirements->setDependencyMap($dependencyMap);
        }

        $this->fillEnabledModulesForCheckRequirements($transitions);
        $this->checkRequirements->setActions([]);
        [$majorVer1, $majorVer2, $minorVersion, $buildVersion] = explode('.', $module->getVersion());

        $this->checkRequirements->addAction(
            ($scenarioModuleState === Module::STATE_ENABLED) ? 'disable' : 'enable',
            $module->getModuleId(),
            $module->isSkin(),
            $module->getMetaData()['canDisable'],
            $module->getMetaData()['minorRequiredCoreVersion']
                ? "{$majorVer1}.{$majorVer2}.{$module->getMetaData()['minorRequiredCoreVersion']}.0"
                : '',
            $module->getMetaData()['dependsOn'] ?? []
        );
    }

    private function fillEnabledModulesForCheckRequirements(array $transitions): void
    {
        $enabledModules   = $this->moduleRepository->getEnabledIds();
        $modulesToEnable  = [];
        $modulesToDisable = [];
        $modulesToRemove = [];

        foreach ($transitions as $moduleId => $transition) {
            if ($transition['stateToSet'] === Module::STATE_ENABLED) {
                $modulesToEnable[] = $moduleId;
            } else {
                $modulesToDisable[] = $moduleId;

                if ($transition['stateToSet'] === Module::STATE_REMOVED) {
                    $modulesToRemove[] = $moduleId;
                }
            }
        }

        $enabledModules = array_diff($enabledModules, $modulesToDisable);
        $enabledModules = array_merge($enabledModules, $modulesToEnable);

        $this->checkRequirements->setEnabledModuleIds($enabledModules);
        $this->checkRequirements->setModulesToRemove($modulesToRemove);
    }

    private function getScenarioState(Module $module, array $transitions): string
    {
        $state = $module->getState();

        return $transitions[$module->getModuleId()]['stateToSet'] ?? $state;
    }

    private function prepareModulesList(array $moduleIds): array
    {
        return array_map(function ($moduleId) {
            /** @var Module $module */
            $module = $this->moduleRepository->findByModuleId($moduleId);

            if (!$module) {
                [$authorName, $moduleName] = explode('-', $moduleId);
            } else {
                $metadata = $module->getMetaData();

                $authorName = $metadata['authorName'];
                $moduleName = $metadata['moduleName'];
            }

            return [
                'id'         => $moduleId,
                'authorName' => $authorName,
                'moduleName' => $moduleName,
            ];
        }, $moduleIds);
    }
}
