<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\DataTransformer;

use App\Domain\XCart;
use App\Entity\Module;
use App\Domain\ModuleDomain;
use App\Repository\ModuleRepository;
use App\Operation\Build\InstallModules\DownloadPacks;
use Doctrine\ORM\EntityManagerInterface;
use ApiPlatform\Core\DataTransformer\DataTransformerInterface;
use ApiPlatform\Core\Exception\InvalidArgumentException;
use Symfony\Component\Validator\Validator\ValidatorInterface;
use Symfony\Component\HttpKernel\Exception\UnprocessableEntityHttpException;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Exception;

class ModuleInputDataTransformer implements DataTransformerInterface
{
    private XCart $XCart;

    private ModuleDomain $moduleDomain;

    private DownloadPacks $downloadPacks;

    private ModuleRepository $moduleRepository;

    private EntityManagerInterface $entityManager;

    private ValidatorInterface $validator;

    public function __construct(
        XCart                  $XCart,
        ModuleDomain           $moduleDomain,
        DownloadPacks          $downloadPacks,
        ModuleRepository       $moduleRepository,
        EntityManagerInterface $entityManager,
        ValidatorInterface     $validator
    ) {
        $this->XCart            = $XCart;
        $this->moduleDomain     = $moduleDomain;
        $this->downloadPacks    = $downloadPacks;
        $this->moduleRepository = $moduleRepository;
        $this->entityManager    = $entityManager;
        $this->validator        = $validator;
    }

    public function transform($object, string $to, array $context = [])
    {
        $violations = $this->validator->validate($object);
        if (count($violations) > 0) {
            throw new InvalidArgumentException(sprintf("Input validations failed: %s", (string) $violations));
        }

        $moduleId = $object->moduleId ?? null;

        if ($moduleId) {
            $module = $this->moduleRepository->findByModuleId($moduleId);

            if (!$module) {
                try {
                    $paths = ($this->downloadPacks)([$moduleId]);

                    foreach ($paths as $moduleId => $path) {
                        $this->moduleDomain->setSourcePath($path);
                        $moduleInfo = $this->moduleDomain->readModuleInfo($moduleId);
                        $module = $this->moduleRepository->createModuleFromArray($moduleInfo);
                        $this->entityManager->persist($module);
                    }

                    $this->entityManager->flush();
                } catch (Exception $e) {
                    throw new UnprocessableEntityHttpException($e->getMessage());
                }
            }
        } else {
            throw new BadRequestHttpException('Module Id is not defined');
        }

        return $module;
    }

    public function supportsTransformation($data, string $to, array $context = []): bool
    {
        if ($data instanceof Module) {
            return false;
        }

        return Module::class === $to && null !== ($context['input']['class'] ?? null);
    }
}
