<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace App\DataTransformer;

use ApiPlatform\Core\DataTransformer\DataTransformerInterface;
use App\DTO\Output\MarketplaceModulesOutput;
use App\Entity\License;
use App\Entity\Module;
use App\Exception\CircularDependencyException;
use App\Marketplace\Marketplace;
use App\Repository\LicenseRepository;

class MarketplaceModuleOutputDataTransformer implements DataTransformerInterface
{
    private Marketplace $marketplace;

    private LicenseRepository $licenseRepository;

    public function __construct(
        Marketplace         $marketplace,
        LicenseRepository   $licenseRepository,
    ) {
        $this->marketplace = $marketplace;
        $this->licenseRepository = $licenseRepository;
    }

    /**
     * @param Module $object
     *
     * @throws CircularDependencyException
     * @throws \JsonException
     */
    public function transform($object, string $to, array $context = []): MarketplaceModulesOutput
    {
        $output = new MarketplaceModulesOutput();
        $output->moduleId = $object->getModuleId();
        $output->version = explode('.', $object->getVersion());
        $output->author = $object->getAuthor();
        $output->name = $object->getName();
        $output->readableName = $object->getMetaData()['moduleName'] ?? '';
        $output->readableAuthor = $object->getMetaData()['authorName'] ?? '';
        $output->minorRequiredCoreVersion = $object->getMetaData()['minorRequiredCoreVersion'] ?? '';
        $output->description = $object->getMetaData()['description'] ?? '';

        $coreLicense = $this->marketplace->getCoreLicense();
        $coreEdition = $coreLicense?->getKeyData()['editionName'] ?? null;
        $output->purchaseUrl = $this->getBuyNowURL($output, $coreLicense);
        $moduleLicense = $this->getModuleLicence($output->author . '-' . $output->name);
        $output->license = $moduleLicense ? $moduleLicense->getKeyValue() : null;
        $message = null;
        if (!empty($output->license)) {
            $message = 'Inactive license key (' . $output->license . ')';
        } else {
            $moduleEditions = [];
            foreach ($module->editions ?? [] as $moduleEdition) {
                if (preg_match('/^(\d+)_(.+)$/', $moduleEdition, $match)) {
                    $moduleEditions[$match[1]] = $match[2];
                }
            }

            if (
                $moduleEditions
                && !in_array($coreEdition, $moduleEditions, true)
            ) {
                $list    = implode(', ', $moduleEditions);
                $message = $coreEdition
                    ? "Does not match license type (requires {$list} edition)"
                    : "Requires {$list} edition";

            }
        }

        if ($message) {
            $output->reason = $message;
        }

        return $output;
    }

    public function supportsTransformation($data, string $to, array $context = []): bool
    {
        return $to === MarketplaceModulesOutput::class && $data instanceof Module;
    }

    public function getBuyNowURL(MarketplaceModulesOutput $module, License $coreLicense)
    {
        $params = [
            'target' => 'generate_invoice',
            'action' => 'buy',
            'proxy_checkout' => 1,
            'inapp_return_url' => '',
            'lickey_1' => md5($coreLicense->getKeyValue()),
        ];

        $httpQuery = $this->buildParamsHttpQuery($params);

        return "https://secure.x-cart.com/customer.php?{$httpQuery}";
    }

    protected function buildParamsHttpQuery(array $params): string
    {
        $urlParams = [];
        foreach ($params as $k => $v) {
            $urlParams[] = $k . '=' . urlencode((string) $v);
        }

        return implode('&', $urlParams);
    }

    private function getModuleLicence(string $moduleId): ?License
    {
        /** @var License $license */
        foreach ($this->licenseRepository->findAll() as $license) {
            if ($license->getModuleId() === $moduleId) {
                $result = $license;
                break;
            }
        }

        return $result ?? null;
    }
}
