<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\DataTransformer;

use ApiPlatform\Core\DataTransformer\DataTransformerInterface;
use App\Domain\XCart;
use App\DTO\Output\LicenseOutput;
use App\Entity\License;
use App\Entity\Module;
use App\Marketplace\Marketplace;
use App\Repository\ModuleRepository;

class LicenseOutputDataTransformer implements DataTransformerInterface
{
    private XCart $XCart;

    private ModuleRepository $moduleRepository;

    private Marketplace $marketplace;

    public function __construct(
        XCart $XCart,
        ModuleRepository $moduleRepository,
        Marketplace $marketplace
    ) {
        $this->XCart            = $XCart;
        $this->moduleRepository = $moduleRepository;
        $this->marketplace      = $marketplace;
    }

    public function transform($object, string $to, array $context = []): LicenseOutput
    {
        /** @var License $object */
        $output                 = new LicenseOutput();
        $output->moduleId       = $object->getModuleId();
        $output->keyValue       = $object->getKeyValue();
        $output->keyType        = $object->getKeyType();
        $output->readableName   = $object->getReadableName();
        $output->prolongKey     = $object->getProlongKey();
        $output->message        = $object->getMessage();
        $output->prolongKey     = $object->getProlongKey();
        $output->registeredAt   = $object->getRegisteredAt();
        $output->expiredAt      = $object->getExpiredAt();
        $output->updatedAt      = $object->getUpdatedAt();
        $output->isTrial        = $object->isTrial();
        $output->isExpired      = $this->isExpired($object);
        $output->isCoreLicense  = $object->isCoreLicense();
        $output->coreVersion    = $this->XCart->getCoreVersion();
        $output->shopIdentifier = $this->marketplace->retrieveStoreIdentity();
        $output->edition        = $object->getEdition();

        if (!$object->isCoreLicense()) {
            /** @var Module $installedModule */
            $installedModule = $this->moduleRepository->findByModuleId($object->getModuleId());

            $output->moduleState = $installedModule
                ? $installedModule->getState()
                : Module::STATE_NOT_INSTALLED;

            $output->moduleSource = $installedModule
                ? $installedModule->getSource()
                : Module::SOURCE_MARKET;

            $output->module = $object->getModule();
        }

        return $output;
    }

    public function supportsTransformation($data, string $to, array $context = []): bool
    {
        return LicenseOutput::class === $to && $data instanceof License;
    }

    private function isExpired(License $object): bool
    {
        $isExpired = $object->isExpired();

        if ($isExpired && !$object->isCoreLicense()) {
            $module            = $object->getModule();
            $marketplaceModule = $this->marketplace->getModule($module['author'], $module['name']);

            $isExpired = !$this->marketplace->isModuleIncludedInCoreEdition($marketplaceModule);
        }

        return $isExpired;
    }
}
