<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\DataProvider;

use ApiPlatform\Core\DataProvider\ItemDataProviderInterface;
use ApiPlatform\Core\DataProvider\RestrictedDataProviderInterface;
use App\Entity\Scenario;
use App\Exception\ClearServiceToolCacheException;
use App\Event\RunnerEvent;
use App\Exception\StepNotReadyException;
use App\Repository\ScenarioRepository;
use App\Output\XCartBufferedOutput;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;
use Throwable;

/**
 * Retrieves scenario by id and run next step of rebuild workflow
 */
final class RebuildScenarioDataProvider implements ItemDataProviderInterface, RestrictedDataProviderInterface
{
    public const OPERATION_NAME = 'rebuild';

    private ScenarioRepository $scenarioRepository;

    private XCartBufferedOutput $bufferedOutput;

    private EventDispatcherInterface $eventDispatcher;

    public function __construct(
        ScenarioRepository       $scenarioRepository,
        XCartBufferedOutput      $bufferedOutput,
        EventDispatcherInterface $eventDispatcher
    ) {
        $this->scenarioRepository = $scenarioRepository;
        $this->bufferedOutput     = $bufferedOutput;
        $this->eventDispatcher    = $eventDispatcher;
    }

    public function supports(string $resourceClass, string $operationName = null, array $context = []): bool
    {
        return $resourceClass === Scenario::class && $operationName === self::OPERATION_NAME;
    }

    /**
     * @param string $id
     *
     * @return Scenario|null
     * @throws Throwable
     */
    public function getItem(string $resourceClass, $id, string $operationName = null, array $context = []): ?object
    {
        /** @var Scenario|null $scenario */
        $scenario = $this->scenarioRepository->findById($id);

        if ($scenario && !$scenario->isDone()) {
            $runnerEvent = new RunnerEvent();
            $runnerEvent
                ->setScenario($scenario)
                ->setOutput($this->bufferedOutput);

            try {
                $this->eventDispatcher->dispatch($runnerEvent, 'service-tool.runner.run-next-step');
            } catch (ClearServiceToolCacheException $e) {
                // After move_packs step a new runtime should start with a new up-to-date cache
                // "die()" is used here because there will be some problems with cache if "return $scenario" will be used (see ECOM-2078)
                die('clearServiceToolCache');
            } catch (StepNotReadyException $e) {
                // Continue, state not changed, next request will try to apply the transition again
            } catch (Throwable $e) {
                $runnerEvent->setError($e->getMessage());
                $this->eventDispatcher->dispatch($runnerEvent, 'service-tool.runner.abort');

                throw $e;
            }

            if ($scenario->isDone()) {
                $this->eventDispatcher->dispatch($runnerEvent, 'service-tool.runner.finish');
            }
        }

        return $scenario;
    }
}
