<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\DataProvider;

use ApiPlatform\Core\DataProvider\CollectionDataProviderInterface;
use ApiPlatform\Core\DataProvider\RestrictedDataProviderInterface;
use App\Entity\License;
use App\Entity\Module;
use App\Marketplace\Marketplace;
use App\Repository\LicenseRepository;
use App\Repository\ModuleRepository;
use Symfony\Component\Serializer\SerializerInterface;

final class MarketplaceDisallowedModulesDataProvider implements CollectionDataProviderInterface, RestrictedDataProviderInterface
{
    public const OPERATION_NAME = 'marketplaceDisallowedModules';

    private Marketplace $marketplace;

    private LicenseRepository $licenseRepository;

    private ModuleRepository $moduleRepository;

    private SerializerInterface $serializer;

    public function __construct(
        Marketplace $marketplace,
        LicenseRepository $licenseRepository,
        ModuleRepository $moduleRepository,
        SerializerInterface $serializer
    ) {
        $this->marketplace       = $marketplace;
        $this->licenseRepository = $licenseRepository;
        $this->moduleRepository  = $moduleRepository;
        $this->serializer        = $serializer;
    }

    public function supports(string $resourceClass, string $operationName = null, array $context = []): bool
    {
        return $resourceClass === Module::class && $operationName === self::OPERATION_NAME;
    }

    public function getCollection(string $resourceClass, string $operationName = null, array $context = []): array
    {
        $result      = [];
        $coreLicense = $this->marketplace->getCoreLicense();

        if (!$coreLicense->isTrial() || $coreLicense->isTrialExpired()) {
            $stateFilterValue = $context['filters']['state'] ?? null;
            $result = $this->getDisallowedModules($stateFilterValue);
        }

        return array_values($result);
    }

    private function getDisallowedModules(?string $stateFilterValue): array
    {
        $resultModules = [];

        array_filter(
            $this->marketplace->getAllModules(),
            function ($marketplaceModule) use ($stateFilterValue, &$resultModules) {
                $localModule = $this->moduleRepository->findByModuleId("{$marketplaceModule['author']}-{$marketplaceModule['name']}");

                if (
                    $localModule
                    && !isset($resultModules[$localModule->getModuleId()])
                    && !$this->marketplace->isModuleIncludedInCoreEdition($marketplaceModule)
                    && !$this->isModuleLicensed($localModule->getModuleId())
                    && in_array($stateFilterValue, [null, $localModule->getState()], true)
                ) {
                    $resultModules[$localModule->getModuleId()] = $localModule;

                    return true;
                }

                return false;
            }
        );

        return array_values($resultModules);
    }

    private function isModuleLicensed(string $moduleId): bool
    {
        return (bool) $this->getModuleLicence($moduleId);
    }

    private function getModuleLicence(string $moduleId): ?License
    {
        /** @var License $license */
        foreach ($this->licenseRepository->findAll() as $license) {
            if ($license->getModuleId() === $moduleId) {
                $result = $license;
                break;
            }
        }

        return $result ?? null;
    }
}
