<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace App\Controller;

use App\Domain\XCart;
use App\Entity\Module;
use App\Marketplace\Marketplace;
use App\Repository\LicenseRepository;
use App\Repository\ModuleRepository;
use Exception;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;

final class VersionsController extends AbstractController
{
    private XCart $XCart;

    private Marketplace $marketplace;

    private ModuleRepository $moduleRepository;

    private LicenseRepository $licenseRepository;

    public function __construct(
        XCart $XCart,
        Marketplace $marketplace,
        ModuleRepository $moduleRepository,
        LicenseRepository $licenseRepository
    ) {
        $this->marketplace       = $marketplace;
        $this->moduleRepository  = $moduleRepository;
        $this->licenseRepository = $licenseRepository;
        $this->XCart             = $XCart;
    }

    private function getModulesData(): array
    {
        $result = [
            'private' => [],
            'public'  => []
        ];

        /** @var Module $module */
        foreach ($this->moduleRepository->getEnabledModules() as $module) {
            $author                = $module->getAuthor();
            $name                  = $module->getName();
            $dynData               = $this->marketplace->getModule($author, $name);
            $moduleType            = (empty($dynData) || (int) $dynData['private']) ? 'private' : 'public';
            $result[$moduleType][] = [
                'service_name'      => $author . '-' . $name,
                'version'           => $module->getVersion(),
                'enabled_timestamp' => $module->getEnabledDate(),
                'author'            => $dynData['readableAuthor'] ?? $author,
                'name'              => $dynData['readableName'] ?? $name,
                'description'       => $dynData['description'] ?? ""
            ];
        }

        return $result;
    }

    public function __invoke(Request $request): array
    {
        $coreLicense = $this->marketplace->getCoreLicense();

        $result = [
            'data' => [
                'store'   => [
                    'license'           => $coreLicense->getEdition() ?? 'Trial',
                    'install_timestamp' => $this->licenseRepository->getXCartInstallationDate(),
                    'version'           => $this->XCart->getCoreVersion(),
                ],
                'modules' => $this->getModulesData()
            ]
        ];

        try {
            $activeSkin = $this->moduleRepository->getSortedEnabledSkinModuleIds()[0] ?? null;
        } catch (Exception $e) {
            $activeSkin = null;
        }

        if ($activeSkin) {
            [ $author, $name ] = explode("-", $activeSkin, 2);
            $dynData = $this->marketplace->getModule($author, $name);
            $result['data']['store']['skin'] = $dynData['readableName'];
        }

        return $result;
    }
}
