<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace App\Controller;

use App\Domain\XCart;
use App\Marketplace\Marketplace;
use Psr\Cache\CacheItemInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Contracts\Cache\CacheInterface;

final class SaveGmvDataController extends AbstractController
{
    public const TTL = 86400; // 1 day

    public function __construct(
        private Marketplace $marketplace,
        private CacheInterface $cache,
        private Filesystem $filesystem,
        private XCart $xcart,
    ) {
    }

    /**
     * @Route("/save-gmv-data/", name="save_gmv_data")
     */
    public function saveGmvData(): JsonResponse
    {
        return new JsonResponse(
            $this->saveGmvDataResult()
        );

    }

    private function saveGmvDataResult(): array
    {
        return $this->cache->get(
            'save_gmv_data',
            function (CacheItemInterface $item) {
                $item->expiresAfter(static::TTL);

                try {
                    $gmvTempFileName = $this->xcart->getGmvPath() . 'gmv_data_tmp.jsonl';

                    if (!$this->filesystem->exists($gmvTempFileName)) {
                        $gmvFileName = $this->xcart->getGmvPath() . 'gmv_data.jsonl';

                        if (!$this->filesystem->exists($gmvFileName)) {
                            return [
                                'error'   => false,
                                'message' => "file {$gmvFileName} does not exist",
                            ];
                        }

                        $this->filesystem->rename($gmvFileName, $gmvTempFileName);
                    }

                    $gmvData = file_get_contents($gmvTempFileName);

                    $response = $this->marketplace->saveGmvData($gmvData);

                    if ($response !== null) {
                        $this->filesystem->remove($gmvTempFileName);
                    }

                    return $response;
                } catch (\Exception $exception) {
                    $item->expiresAfter(0);

                    return [
                        'error'   => true,
                        'message' => $exception->getMessage(),
                    ];
                }
            },
        );
    }
}
