<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace App\Controller;

use App\Domain\XCart;
use App\Entity\ModuleUpgrade;
use App\Exception\GetUpgradesException;
use App\Marketplace\Marketplace;
use App\Operation\Build\Upgrade\CheckUpgradeRequirements;
use App\Repository\ModuleRepository;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;

final class ModuleUpgradesController extends AbstractController
{
    private Marketplace $marketplace;

    private ModuleRepository $moduleRepository;

    private CheckUpgradeRequirements $checkUpgradeRequirements;

    private string $packsDir;

    public function __construct(
        XCart                    $XCart,
        Marketplace              $marketplace,
        ModuleRepository         $moduleRepository,
        CheckUpgradeRequirements $checkUpgradeRequirements
    ) {
        $this->packsDir                 = $XCart->getPacksPath();
        $this->marketplace              = $marketplace;
        $this->moduleRepository         = $moduleRepository;
        $this->checkUpgradeRequirements = $checkUpgradeRequirements;
    }

    /**
     * @throws GetUpgradesException
     * @throws \Exception
     */
    public function __invoke(Request $request): array
    {
        $moduleUpgrades = [];

        $withAdditionalInfo = filter_var(
            $request->query->get('withAdditionalInfo', false),
            FILTER_VALIDATE_BOOL
        );

        foreach ($this->marketplace->getUpgrades($withAdditionalInfo) as $upgradeType => $upgrades) {
            foreach ($upgrades as $moduleId => $upgrade) {
                if (Marketplace::CORE_MODULE_ID === $moduleId) {
                    $failedPhpTest                = $this->checkUpgradeRequirements->phpVersionCheck([Marketplace::CORE_MODULE_ID => $upgrade]);
                    $upgrade['readableAuthor']    = 'X-Cart team';
                    $upgrade['readableName']      = 'Core';
                    $upgrade['incompatibleError'] = $failedPhpTest
                        ? [
                            'label'  => 'upgrade-details-page.requirements-warning.php.message',
                            'params' => [$failedPhpTest['coreVersionTo'], $failedPhpTest['phpMinRequiredVersion'], phpversion()]]
                        : [];
                } else {
                    $upgrade['incompatibleError'] = [];
                }

                $moduleUpgrades[$moduleId][$upgradeType] = $upgrade;
            }
        }

        $result = [];
        foreach ($moduleUpgrades as $moduleId => $upgrades) {
            $result[] = new ModuleUpgrade(
                $moduleId,
                $upgrades,
                $this->getModuleData($moduleId)
            );
        }

        return $result;
    }

    private function getPackagePath(string $moduleId, array $upgrade): string
    {
        $upgradeVersion = Marketplace::implodeVersion($upgrade['version']);

        return "{$this->packsDir}{$moduleId}-v{$upgradeVersion}";
    }

    private function getModuleData(string $moduleId): array
    {
        $enabled = in_array($moduleId, $this->moduleRepository->getEnabledIds(), true);

        return [
            'enabled' => Marketplace::CORE_MODULE_ID === $moduleId ? true : $enabled,
        ];
    }
}
