<?php

namespace App\Controller;

use App\Deployment\Runner;
use App\Domain\ModuleDomain;
use App\Domain\ShopURLs;
use App\Entity\Scenario;
use App\Event\RunnerEvent;
use App\Exception\BuildException;
use App\Exception\CheckPermissionsException;
use App\Exception\DownloadModulesException;
use App\Exception\DownloadPackException;
use App\Marketplace\Marketplace;
use App\Marketplace\MarketplaceStorage;
use App\Operation\Build\GenerateTransitions;
use App\Operation\Build\InstallModules\CheckPermissions;
use App\Operation\Build\InstallModules\DownloadPacks;
use App\Operation\Build\InstallModules\MoveInstallPacks;
use App\Repository\ModuleRepository;
use Doctrine\ORM\EntityManagerInterface;
use MJS\TopSort\CircularDependencyException;
use MJS\TopSort\ElementNotFoundException;
use Psr\Log\LoggerInterface;
use SodiumException;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

final class MarketModuleInstall extends AbstractController
{
    protected ModuleRepository $moduleRepository;

    protected ShopURLs $shopURLs;

    protected Marketplace $marketplace;

    protected MarketplaceStorage $storage;

    private ModuleDomain $moduleDomain;

    private Runner $runner;

    private GenerateTransitions $generateTransitions;

    private EntityManagerInterface $entityManager;

    private EventDispatcherInterface $eventDispatcher;

    private DownloadPacks $downloadPacks;

    private MoveInstallPacks $moveInstallPacks;

    private CheckPermissions $checkPermissions;

    private LoggerInterface $logger;

    private $config;

    public function __construct(
        $config,
        Runner $runner,
        GenerateTransitions $generateTransitions,
        DownloadPacks $downloadPacks,
        MoveInstallPacks $moveInstallPacks,
        CheckPermissions $checkPermissions,
        EntityManagerInterface $entityManager,
        EventDispatcherInterface $eventDispatcher,
        LoggerInterface $logger,
        ModuleRepository $moduleRepository,
        ModuleDomain $moduleDomain,
        Marketplace $marketplace,
        MarketplaceStorage $storage,
        ShopURLs $shopURLs
    ) {
        $this->config              = $config;
        $this->runner              = $runner;
        $this->generateTransitions = $generateTransitions;
        $this->entityManager       = $entityManager;
        $this->eventDispatcher     = $eventDispatcher;
        $this->downloadPacks       = $downloadPacks;
        $this->moveInstallPacks    = $moveInstallPacks;
        $this->checkPermissions    = $checkPermissions;
        $this->logger              = $logger;
        $this->moduleRepository    = $moduleRepository;
        $this->moduleDomain        = $moduleDomain;
        $this->marketplace         = $marketplace;
        $this->storage             = $storage;
        $this->shopURLs            = $shopURLs;
    }

    /**
     * @Route("/market_module_installer/", name="app_market_module_install")
     */
    public function market_module_installer(Request $request): Response
    {
        $modules = $this->getModulesToInstall($request);
        // relative URL is used to prevent problems with proxies XCB-1481
        $returnUrl = $this->shopURLs->getShopWebDir() . "/admin/?target=apps#/installed-addons";

        if (!$modules['toInstall'] && !$modules['toEnable']) {
            $this->logger->info('Nothing to install_1', [print_r($request->getRequestUri(), true)]);

            return $this->redirect($returnUrl);
        }

        $scenario = new Scenario();
        $scenario->setType(Scenario::TYPE_REBUILD);

        if ($this->isRequestedDelayedInstall($request)) {
            // This is a temp runtime scenario. Avoid creating var/.rebuildInProgress in \App\EventListener\RunnerListener::start
            $scenario->setState(Scenario::STATE_IN_PROGRESS);
        } else {
            // Scenario is to be written to DB. Init scenario via \App\EventListener\RunnerListener::start
            $runnerEvent = new RunnerEvent();
            $runnerEvent->setScenario($scenario);
            $this->eventDispatcher->dispatch($runnerEvent, 'service-tool.runner.start');
        }

        if ($modules['toInstall']) {
            try {
                $packPaths = ($this->downloadPacks)($modules['toInstall']);
                ($this->checkPermissions)($modules['toInstall']);

                foreach ($packPaths as $moduleId => $path) {
                    $bkpSourcePath = $this->moduleDomain->getSourcePath();
                    $this->moduleDomain->setSourcePath($path);
                    $moduleInfo = $this->moduleDomain->readModuleInfo($moduleId);
                    $module     = $this->moduleRepository->createModuleFromArray($moduleInfo);
                    $this->entityManager->persist($module);
                    $this->moduleDomain->setSourcePath($bkpSourcePath);
                }

                $this->entityManager->flush();
            } catch (CheckPermissionsException|DownloadModulesException|DownloadPackException $e) {
                // to avoid 500 error for customer
                $this->logger->error($e->getMessage(), ['exception_1' => $e, $e->getTraceAsString(), "toInstall:" . print_r($modules['toInstall'], true)]);
                if (!empty($runnerEvent)) {
                    // delete var/.rebuildInProgress
                    $this->eventDispatcher->dispatch($runnerEvent, 'service-tool.runner.abort');
                }

                return $this->redirect($returnUrl . '#err_code=Id=' . $this->getErrorCodeId($e));
            }

            if ($this->isRequestedDelayedInstall($request)) {
                ($this->moveInstallPacks)($packPaths);
            } else {
                $scenario->setMetaData(array_merge($scenario->getMetaData(), ['packPaths' => $packPaths]));
            }
        }

        try {
            $scenario->setTransitions(
                ($this->generateTransitions)(
                    $modules['toEnable'],
                    $modules['toDisable'],
                    [], // nothing to remove, confliting modules will be just disabled
                    $modules['toInstall']
                )
            );
            if (!$this->isRequestedDelayedInstall($request)) {
                $this->entityManager->persist($scenario);
                $this->entityManager->flush();
            }
        } catch (BuildException|CircularDependencyException|ElementNotFoundException $e) {
            // to avoid 500 error for customer
            $this->logger->error($e->getMessage(), ['exception_2' => $e, $e->getTraceAsString(), print_r($modules, true)]);
            if (!empty($runnerEvent)) {
                // delete var/.rebuildInProgress
                $this->eventDispatcher->dispatch($runnerEvent, 'service-tool.runner.abort');
            }

            return $this->redirect($returnUrl . '#err_code=Id=' . $this->getErrorCodeId($e));
        }

        // relative URL is used to prevent problems with proxies XCB-1481
        if ($scenario->getTransitions()) {
            $shopUrlParam = $this->shopURLs->getShopWebDir() ? "&xcUrl={$this->shopURLs->getShopURL()}" : '';

            $returnUrl = $this->isRequestedDelayedInstall($request)
            && ($idsForReturnUrl = $this->getIdsForReturnUrl($modules))
                ? $returnUrl . '#modules2enable=' . $idsForReturnUrl
                : $this->shopURLs->getShopWebDir() . "/rebuild.html?scenarioId={$scenario->getId()}{$shopUrlParam}&returnURL=" . urlencode($returnUrl . "#scenario=Id={$scenario->getId()}");
        } else {
            $this->logger->info('Nothing to install_2', [print_r($request->getRequestUri(), true)]);
        }

        return $this->redirect($returnUrl);
    }

    private function getModulesToInstall(Request $request): array
    {
        $modulesToInstall = $modulesToDisable = $modulesToEnable = [];
        $getParams        = $request->query->all();
        $target           = $getParams['target'] ?? '';
        if ($target === 'market_install_module') {
            $mainInstall = $this->parseModuleId($getParams['mainInstall'] ?? '');
            if ($mainInstall) {
                foreach (($getParams['installTip'] ?? []) as $moduleStr) {
                    $modulesToInstall[] = $this->parseModuleId($moduleStr);
                }
                foreach (($getParams['disableTip'] ?? []) as $moduleStr) {
                    $modulesToDisable[] = $this->parseModuleId($moduleStr);
                }
                // The order of modules matters. Dependencies must be listed before the target module
                $modulesToInstall[] = $mainInstall;
            }
        } elseif ($target === 'market_successful_payment') {
            $modulesToInstall = $this->getPaidModulesToInstallByToken($getParams['encryptedTokens'] ?? '');
        }

        foreach ($modulesToInstall as $key => $moduleId) {
            $module = $this->moduleRepository->findByModuleId($moduleId);
            if ($module) {
                // The module is local check if we have just to enable it
                if (!$module->isEnabled()) {
                    $modulesToEnable[] = $moduleId;
                }
                unset($modulesToInstall[$key]);
            }
        }

        foreach ($modulesToDisable as $key => $moduleId) {
            $module = $this->moduleRepository->findByModuleId($moduleId);
            if (!$module || !$module->isEnabled()) {
                // The module is already disabled
                unset($modulesToDisable[$key]);
            }
        }

        return ['toInstall' => $modulesToInstall, 'toEnable' => $modulesToEnable, 'toDisable' => $modulesToDisable];
    }

    private function getPaidModulesToInstallByToken(string $encryptedXbTokens): array
    {
        if (empty($encryptedXbTokens)) {
            return [];
        }
        try {
            // see https://www.php.net/manual/en/function.sodium-crypto-box-seal.php
            $keypair           = $this->storage->getValue('secret_public_keypair') ?: '';
            $keypair           = sodium_base642bin($keypair, SODIUM_BASE64_VARIANT_URLSAFE_NO_PADDING);
            $encryptedXbTokens = sodium_base642bin($encryptedXbTokens, SODIUM_BASE64_VARIANT_URLSAFE_NO_PADDING);
            $xbTokens          = sodium_crypto_box_seal_open($encryptedXbTokens, $keypair);
        } catch (SodiumException $e) {
            $this->logger->error($e->getMessage(), ['exception' => $e, $e->getTraceAsString(), 'toDecrypt' => print_r($encryptedXbTokens, true)]);
        }
        if (empty($xbTokens)) {
            $this->logger->error(__METHOD__ . ': xbTokens were not decrypted', ['toDecrypt' => print_r($encryptedXbTokens, true)]);

            return [];
        }
        $xbTokens = unserialize($xbTokens) ?? [];

        $modules2install = [];

        foreach ($xbTokens as $xbToken => $_i) {
            foreach ($this->marketplace->retrieveLicensesByToken($xbToken) as $licenceType => $licenceKeys) {
                if (!$licenceKeys || $licenceType === 'prolongation') {
                    continue;
                }
                foreach ($licenceKeys as $licenceKey) {
                    if ($licenceObj = $this->marketplace->registerLicense($licenceKey)) {
                        $moduleId = $licenceObj->getModuleId();
                        if (!in_array($moduleId, $modules2install)) {
                            $modules2install[] = $moduleId;
                        }
                    }
                }
            }
        }

        return $modules2install ?? [];
    }

    private function isRequestedDelayedInstall(Request $request): bool
    {
        return (bool)($request->query->get('delay_install') ?? false);
    }

    private function parseModuleId($stringWithModuleId): string
    {
        $arrWithModuleId = explode('::', $stringWithModuleId) ?? [];

        return str_replace('/', '-', $arrWithModuleId[0] ?? '');
    }

    /**
     * Generate a string like
     * urlSearchParams = new URLSearchParams('QSL-AbandonedCartReminder=1&QSL-AbandonedCartReminder2=0');Object.fromEntries(urlSearchParams.entries());
     * Used in src/assets/web/admin/service/src/utils.js
     */
    private function getIdsForReturnUrl(array $modules): string
    {
        $result = [
            array_fill_keys($modules['toInstall'] ?? [], 1),
            array_fill_keys($modules['toEnable'] ?? [], 1),
            array_fill_keys($modules['toDisable'] ?? [], 0),
        ];

        return http_build_query(array_merge(...$result));
    }

    private function getErrorCodeId($exception): string
    {
        $errorCode = get_class($exception);
        if ($pos = strrpos($errorCode, '\\')) {
            $errorCode = substr($errorCode, $pos + 1);
        }

        if (
            $exception instanceof BuildException
            && ($reasonCode = $exception->getLastReasonCode())
            && in_array($reasonCode, $this->supportedServiceUITranslations())
        ) {
            $errorCode = str_replace('BuildException', 'BE', $errorCode);
            $errorCode .= ".{$reasonCode}"; // example 'scenario.transition.module.err.BE.conflict-with-disable'
        }

        return $errorCode;
    }

    /**
     * Sync with \XLite\View\ServiceUI::getCommentedData()
     * and
     * Sync with constants values in \App\Entity\CheckRequirementsResult class
     */
    private function supportedServiceUITranslations(): array
    {
        return [
            \App\Entity\CheckRequirementsResult::CANT_DISABLE_BY_BACK_DEPENDENCY_REASON => 'cant-disable-by-back-dependency',
            \App\Entity\CheckRequirementsResult::CANT_DISABLE_BY_MODULE_PROPERTY_REASON => 'cant-disable-by-module-property',
            \App\Entity\CheckRequirementsResult::CANT_ENABLE_BY_DEPENDENCY_REASON       => 'cant-enable-by-dependency',
            \App\Entity\CheckRequirementsResult::CANT_ENABLE_BY_INCOMPATIBLE_REASON     => 'cant-enable-by-incompatible',
            \App\Entity\CheckRequirementsResult::CANT_ENABLE_BY_VERSION_REASON          => 'cant-enable-by-minorRequiredCoreVersion',
            \App\Entity\CheckRequirementsResult::CONFLICT_WITH_DISABLE_REASON           => 'conflict-with-disable',
            \App\Entity\CheckRequirementsResult::CONFLICT_WITH_ENABLE_REASON            => 'conflict-with-enable',
        ];
    }
}
