<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace App\Controller;

use App\Domain\XCart;
use App\Entity\IntegrityViolation;
use App\Exception\GetHashException;
use App\Marketplace\Marketplace;
use App\Operation\Integrity\ComparedFileRetriever;
use App\Operation\Integrity\CoreHashRetriever;
use App\Operation\Integrity\GenerateViolations;
use App\Operation\Integrity\ModuleHashRetriever;
use App\Repository\ModuleRepository;
use App\Repository\ScenarioRepository;
use Exception;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Uid\UuidV4;

final class IntegrityViolationsController extends AbstractController
{
    private XCart $XCart;

    private CoreHashRetriever $coreHashRetriever;

    private ModuleHashRetriever $moduleHashRetriever;

    private ComparedFileRetriever $comparedFileRetriever;

    private GenerateViolations $generateViolations;

    private ModuleRepository $moduleRepository;

    private Marketplace $marketplace;

    public function __construct(
        XCart $XCart,
        CoreHashRetriever $coreHashRetriever,
        ModuleHashRetriever $moduleHashRetriever,
        ComparedFileRetriever $comparedFileRetriever,
        GenerateViolations $generateViolations,
        Marketplace $marketplace,
        ModuleRepository $moduleRepository
    ) {
        $this->XCart                 = $XCart;
        $this->coreHashRetriever     = $coreHashRetriever;
        $this->moduleHashRetriever   = $moduleHashRetriever;
        $this->comparedFileRetriever = $comparedFileRetriever;
        $this->generateViolations    = $generateViolations;
        $this->marketplace           = $marketplace;
        $this->moduleRepository      = $moduleRepository;
    }

    /**
     * @throws GetHashException
     * @throws Exception
     */
    public function __invoke(
        Request $request,
        ScenarioRepository $scenarioRepository
    ): array {
        $scenarioId         = new UuidV4($request->get('scenarioId'));
        $scenario           = $scenarioRepository->findById($scenarioId);
        $scenarioMetadata   = $scenario->getMetaData();
        $modulesToUpgrade   = $scenarioMetadata['modulesToUpgrade'] ?? [];
        $moduleIdsToUpgrade = array_keys($modulesToUpgrade);

        $coreIntegrityViolations = (in_array(Marketplace::CORE_MODULE_ID, $moduleIdsToUpgrade, true))
            ? $this->getCoreIntegrityViolations()
            : [];

        return array_merge(
            $coreIntegrityViolations,
            $this->getModulesIntegrityViolations($moduleIdsToUpgrade)
        );
    }

    /**
     * @throws GetHashException
     */
    private function getCoreIntegrityViolations(): array
    {
        $coreKnown = ($this->coreHashRetriever)(
            MarketPlace::explodeVersion($this->XCart->getCoreVersion())
        );

        if (isset($coreKnown['message'])) {
            throw GetHashException::fromGetCoreHashResponseWithError($coreKnown['message']);
        }

        $coreActual          = ($this->comparedFileRetriever)($this->XCart->getSourcePath(), array_keys($coreKnown));
        $integrityViolations = ($this->generateViolations)($coreKnown, $coreActual);
        $result              = [];

        $integrityViolations = array_filter(
            $integrityViolations,
            fn ($violation) => !preg_match($this->getExcludeFilePattern(), $violation['filepath'])
        );

        foreach ($integrityViolations as $integrityViolation) {
            if (in_array($integrityViolation['type'], ['modified', 'removed'])) {
                $result[] = new IntegrityViolation($integrityViolation['filepath']);
            }
        }

        return $result;
    }

    private function getExcludeFilePattern(): string
    {
        $patterns = [
            ".*vendor\/composer\/.*"
        ];

        return '/^(?:' . implode('|', $patterns) . ')/Ss';
    }

    /**
     * @throws GetHashException
     * @throws Exception
     */
    private function getModulesIntegrityViolations(array $moduleIdsToUpgrade): array
    {
        $moduleVersionHashes = array_filter(
            $this->marketplace->getInstalledModuleVersionHashes(),
            static fn ($moduleId) => in_array($moduleId, $moduleIdsToUpgrade, true),
            ARRAY_FILTER_USE_KEY
        );

        $moduleHash = ($this->moduleHashRetriever)($moduleVersionHashes);
        $result     = [];

        foreach ($moduleHash as $moduleId => $data) {
            $known = $data['result'] ?? null;
            if ($known) {
                $actual = $this->getModuleCurrentVersionHashes($moduleId);

                // For the delayed upgrade $actual will be equal to empty array.
                // Integrity violations should not be shown for the delayed upgrade.
                if ($actual) {
                    $moduleIntegrityViolations = ($this->generateViolations)($known, $actual);

                    foreach ($moduleIntegrityViolations as $integrityViolation) {
                        if (in_array($integrityViolation['type'], ['modified', 'removed'])) {
                            $result[] = new IntegrityViolation($integrityViolation['filepath']);
                        }
                    }
                }
            } else {
                throw GetHashException::fromGetAddonHashBatchResponseWithError($moduleId, $data['error']);
            }
        }

        return $result;
    }

    /**
     * @throws GetHashException
     */
    private function getModuleCurrentVersionHashes(string $moduleId): array
    {
        $actual = [];
        $module = $this->moduleRepository->findByModuleId($moduleId);
        $actualModuleVersionHash = ($this->moduleHashRetriever)(
            [$moduleId => Marketplace::getModuleVersionHash($module->getAuthor(), $module->getName(), $module->getVersion())]
        );

        if (!empty($actualModuleVersionHash)) {
            $actual = $actualModuleVersionHash[$moduleId]['result'] ?? null;
        }

        if ($actual === null) {
            throw GetHashException::fromGetAddonHashBatchResponseWithError($moduleId, $actualModuleVersionHash[$moduleId]['error']);
        }

        return $actual;
    }
}
