<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace App\Controller;

use App\Domain\XCart;
use App\Entity\IncorrectPermissionsCommand;
use App\Repository\ScenarioRepository;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Uid\UuidV4;

final class IncorrectPermissionsCommandController extends AbstractController
{
    private string $sourcePath;

    public function __construct(XCart $XCart)
    {
        $this->sourcePath = $XCart->getSourcePath();
    }

    public function __invoke(
        Request $request,
        ScenarioRepository $scenarioRepository
    ): array {
        $scenarioId = new UuidV4($request->get('scenarioId'));

        $scenario         = $scenarioRepository->findById($scenarioId);
        $scenarioMetadata = $scenario->getMetaData();
        $upgradeEntries   = $scenarioMetadata['upgradeEntries'] ?? [];

        return $this->getIncorrectPermissionsCommands($upgradeEntries);
    }

    private function getIncorrectPermissionsCommands(array $upgradeEntries): array
    {
        $result = [];
        $incorrectPermissions = [];

        foreach ($upgradeEntries as $moduleUpgradeEntries) {
            foreach ($moduleUpgradeEntries as $upgradeEntry) {
                $upgradeEntry['filepath'] = "{$this->sourcePath}{$upgradeEntry['filepath']}";

                if (!$this->isEntryValid($upgradeEntry)) {
                    $path = $upgradeEntry['filepath'];
                    $type = is_dir($path) ? 'directories' : 'files';

                    $commonPath = $this->getCommonPath($path);
                    if ($commonPath) {
                        if (!isset($result["{$commonPath}-{$type}"])) {
                            $cmd = is_dir($path)
                                ? 'find ' . $commonPath . ' -type d -execdir chmod 777 "{}" \\;'
                                : 'find ' . $commonPath . ' -type f -execdir chmod 666 "{}" \\;';

                            $result["{$commonPath}-{$type}"] = new IncorrectPermissionsCommand($cmd);
                        }
                    } else {
                        $incorrectPermissions[$type][] = $path;
                    }
                }
            }
        }

        foreach ($incorrectPermissions as $type => $paths) {
            $permission = ($type === 'directories') ? '777' : '666';

            $result[] = new IncorrectPermissionsCommand(
                'chmod ' . $permission . ' ' . implode(' ', $paths) . ';'
            );
        }

        return array_values($result);
    }

    private function getCommonPath(string $path): string
    {
        $commonPaths = [
            'classes',
            'Includes',
            'sql',
            'templates',
            'assets',
            'modules',
            'service-tool'
        ];

        foreach ($commonPaths as $commonPath) {
            if (strpos($path, $this->sourcePath . $commonPath) === 0) {
                return $this->sourcePath . $commonPath;
            }
        }

        return '';
    }

    private function isEntryValid(array $moduleFileToMove): bool
    {
        $type = $moduleFileToMove['type'];
        $path = $moduleFileToMove['filepath'];

        if ($type === 'removed') {
            return !file_exists($path) || is_writable($path);
        }

        if ($type === 'modified' || $type === 'added') {
            if (file_exists($path)) {
                return is_writable($path);
            }

            return $this->isFirstExistedDirIsWritable($path);
        }

        return false;
    }

    private function isFirstExistedDirIsWritable(string $path): bool
    {
        while ($dir = dirname($path)) {
            if (is_dir($dir)) {
                return is_writable($dir);
            }
        }

        return false;
    }
}
