<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace App\Controller;

use App\Domain\XCart;
use App\DTO\Output\ScenarioDownloadPacksProgressOutput;
use App\Entity\Scenario;
use App\Exception\DownloadPackException;
use App\Exception\PartialDownloadPackException;
use App\Exception\RetrieveSingleChunkException;
use App\Marketplace\Marketplace;
use App\Operation\Build\ExtractPack;
use App\Operation\Download\DownloaderInterface;
use App\Repository\ScenarioRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Uid\UuidV4;

final class DownloadPacksController extends AbstractController
{
    private int $chunkSize;

    private string $packsDir;

    private DownloaderInterface $downloader;

    private Scenario $scenario;

    private EntityManagerInterface $entityManager;

    private ExtractPack $extractPack;

    public function __construct(
        int $chunkSize,
        DownloaderInterface $downloader,
        EntityManagerInterface $entityManager,
        ExtractPack $extractPack,
        XCart $XCart
    ) {
        $this->chunkSize     = $chunkSize;
        $this->downloader    = $downloader;
        $this->entityManager = $entityManager;
        $this->extractPack   = $extractPack;
        $this->packsDir      = $XCart->getPacksPath();
    }

    /**
     * @noinspection PhpRedundantCatchClauseInspection
     * @throws DownloadPackException
     * @throws PartialDownloadPackException
     */
    public function __invoke(
        Request $request,
        ScenarioRepository $scenarioRepository,
        Filesystem $filesystem
    ): ScenarioDownloadPacksProgressOutput {
        $scenarioId            = new UuidV4($request->get('id'));
        $this->scenario        = $scenarioRepository->findById($scenarioId);
        $scenarioMetadata      = $this->scenario->getMetaData();
        $downloadPacksProgress = $scenarioMetadata['downloadPacksProgress'] ?? [];

        if (!$downloadPacksProgress) {
            $this->prepareScenario();
        } else {
            $modulesToUpgrade  = $scenarioMetadata['modulesToUpgrade'];
            $moduleId          = $downloadPacksProgress['moduleId'];
            $moduleUpgradeData = $modulesToUpgrade[$moduleId];

            try {
                $this->downloader->download(
                    $moduleId,
                    $moduleUpgradeData,
                    "{$downloadPacksProgress['packagePath']}.tar.gz",
                    $downloadPacksProgress['request']['target'],
                    $downloadPacksProgress['request']['params'],
                    $downloadPacksProgress['sizePos'],
                    $downloadPacksProgress['totalSize']
                );

                $this->prepareScenarioForNextPack();
            } catch (RetrieveSingleChunkException $e) {
                $this->prepareScenarioForNextChunk();
            }
        }

        return $this->getOutput();
    }

    private function prepareScenario(): void
    {
        $scenarioMetadata = $this->scenario->getMetaData();

        $scenarioMetadata['downloadPacksProgress'] = $this->getDownloadPacksProgress(0);
        $this->scenario->setMetaData($scenarioMetadata);
        $this->entityManager->flush();
    }

    private function prepareScenarioForNextChunk(): void
    {
        $scenarioMetadata = $this->scenario->getMetaData();

        $scenarioMetadata['downloadPacksProgress']['sizePos'] += $this->chunkSize;
        $this->scenario->setMetaData($scenarioMetadata);
        $this->entityManager->flush();
    }

    private function prepareScenarioForNextPack(): void
    {
        $scenarioMetadata = $this->scenario->getMetaData();

        ($this->extractPack)($scenarioMetadata['downloadPacksProgress']['packagePath']);

        $newCurrentPackIndex = $scenarioMetadata['downloadPacksProgress']['index'] + 1;

        $scenarioMetadata['downloadPacksProgress'] = $this->getDownloadPacksProgress($newCurrentPackIndex);
        $this->scenario->setMetaData($scenarioMetadata);
        $this->entityManager->flush();
    }

    private function getDownloadPacksProgress(int $currentPackIndex): array
    {
        $scenarioMetadata = $this->scenario->getMetaData();
        $modulesToUpgrade = $scenarioMetadata['modulesToUpgrade'];
        $packsCount       = count($modulesToUpgrade);

        if ($currentPackIndex < $packsCount) {
            $packsToDownload   = array_keys($modulesToUpgrade);
            $moduleId          = $packsToDownload[$currentPackIndex];
            $moduleUpgradeData = $modulesToUpgrade[$moduleId];
            $readableName      = $moduleUpgradeData['readableName'];
            $upgradeVersion    = Marketplace::implodeVersion($moduleUpgradeData['version']);
            $packagePath       = "{$this->packsDir}{$moduleId}-v{$upgradeVersion}";
            $target            = $this->downloader->getTarget($moduleId);
            $params            = $this->downloader->getParams($moduleId, $moduleUpgradeData);
            $totalSize         = $this->downloader->getSize($target, $params, $moduleId);
        }

        return [
            'index'        => $currentPackIndex,
            'packsCount'   => $packsCount,
            'moduleId'     => $moduleId ?? '',
            'readableName' => $readableName ?? '',
            'sizePos'      => 0,
            'totalSize'    => $totalSize ?? 0,
            'packagePath'  => $packagePath ?? '',
            'request'      => [
                'target' => $target ?? '',
                'params' => $params ?? [],
            ],
        ];
    }

    private function getOutput(): ScenarioDownloadPacksProgressOutput
    {
        $downloadPacksProgress = $this->scenario->getMetaData()['downloadPacksProgress'];

        if ($downloadPacksProgress['index'] < $downloadPacksProgress['packsCount']) {
            $status  = ScenarioDownloadPacksProgressOutput::STATUS_IN_PROGRESS;
            $message = $this->getDownloadPacksProgressMessage();
        } else {
            $status  = ScenarioDownloadPacksProgressOutput::STATUS_DONE;
            $message = '';
        }

        return new ScenarioDownloadPacksProgressOutput($status, $message);
    }

    private function getDownloadPacksProgressMessage(): string
    {
        $scenarioMetadata         = $this->scenario->getMetaData();
        $downloadPacksProgress    = $scenarioMetadata['downloadPacksProgress'];
        $currentPackIndexReadable = $downloadPacksProgress['index'] + 1;
        $currentPackSizePos       = $this->convertToMb(min($downloadPacksProgress['sizePos'], $downloadPacksProgress['totalSize']));
        $currentPackTotalSize     = $this->convertToMb($downloadPacksProgress['totalSize']);

        return "Download {$downloadPacksProgress['readableName']} ({$currentPackIndexReadable} of {$downloadPacksProgress['packsCount']} packs). {$currentPackSizePos}/{$currentPackTotalSize} MB is loaded.";
    }

    private function convertToMb($size): float
    {
        return round($size / 1024 / 1024, 1);
    }
}
