<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Command;

use App\Domain\XCart;
use App\Entity\Module;
use App\Marketplace\Marketplace;
use App\Repository\LicenseRepository;
use App\Repository\ModuleRepository;
use Exception;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Helper\Table;
use Symfony\Component\Console\Helper\TableCell;
use Symfony\Component\Console\Helper\TableCellStyle;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

final class VersionInfoCommand extends Command
{
    public const DESCRIPTION_COLUMN_WIDTH = 160;

    protected static $defaultName = 'xcst:version-info';

    private XCart $XCart;

    private Marketplace $marketplace;

    private ModuleRepository $moduleRepository;

    private LicenseRepository $licenseRepository;

    public function __construct(
        XCart $XCart,
        Marketplace $marketplace,
        ModuleRepository $moduleRepository,
        LicenseRepository $licenseRepository
    ) {
        parent::__construct();

        $this->XCart             = $XCart;
        $this->marketplace       = $marketplace;
        $this->moduleRepository  = $moduleRepository;
        $this->licenseRepository = $licenseRepository;
    }

    protected function configure()
    {
        $this->setDescription('Prints the current build info.');
    }

    /**
     * @throws Exception
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);

        $io->title('Store info');

        $coreLicense = $this->marketplace->getCoreLicense();
        $licenseVersion = $coreLicense ? $coreLicense->getEdition() : 'Trial';
        $activeSkin = $this->moduleRepository->getSortedEnabledSkinModuleIds()[0] ?? null;
        $installDate = $this->licenseRepository->getXCartInstallationDate();
        $installDate = date('d M Y H:i', $installDate);

        $io->writeln("<info>Version:</info> <comment>{$this->XCart->getCoreVersion()}</comment>");
        $io->writeln("<info>Edition:</info> <comment>{$licenseVersion}</comment>");
        $io->writeln("<info>Skin:</info> <comment>{$activeSkin}</comment>");
        $io->writeln("<info>Installation date:</info> <comment>{$installDate}</comment>");

        [$table1, $table2] = $this->getEnabledModulesTables($output);

        $io->newLine();
        $io->writeln("<fg=blue>Private and custom modules (enabled):</>");
        $table1->render();

        $io->newLine();
        $io->writeln("<fg=blue>Public installed modules (enabled):</>");
        $table2->render();

        $io->newLine();

        return Command::SUCCESS;
    }

    private function getEnabledModulesTables(OutputInterface $output): array
    {
        $customAndPrivateModulesTable = new Table($output);
        $publicModulesTable = new Table($output);

        $cols = ['Module Id', 'Version', 'Installation date'];
        $customAndPrivateModulesTable->setHeaders($cols);
        $publicModulesTable->setHeaders($cols);

        foreach ($this->moduleRepository->getEnabledModules() as $module) {
            /** @var Module $module */
            $dynamicData = $this->marketplace->getModule($module->getAuthor(), $module->getName());

            $name = implode('-', [$module->getAuthor(), $module->getName()]);

            if (empty($dynamicData) || (int) $dynamicData['private']) {
                $customAndPrivateModulesTable->addRow($this->getCellData($module, $name));
            } else {
                $publicModulesTable->addRow($this->getCellData($module, $name));
            }
        }

        return [$customAndPrivateModulesTable, $publicModulesTable];
    }

    private function getCellData(Module $module, $name): array
    {
        return [$name, $module->getVersion(), date('d M Y H:i', $module->getEnabledDate())];
    }
}
