<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Command;

use App\Entity\ConfigurationFile;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

final class VerifyConfigCommand extends Command
{
    /**
     * @var string
     */
    protected static $defaultName = 'xcst:verify-config';

    protected function configure()
    {
        $help = <<< HELP
<info>Options:</info>
    <fg=red;bg=gray;options=bold>--buildName</>, <fg=red;bg=gray;options=bold>-b</>   - The build specification (the list of modules and default data). The build configurations are stored as yaml files in "service-tool/resources". Specify the "service-tool/resources" related file path with no extension., e.g., "build/default"
    <fg=red;bg=gray;options=bold>--language</>, <fg=red;bg=gray;options=bold>-l</>    - The build language. The value is used for a localized package name.
    <fg=red;bg=gray;options=bold>--development</>, <fg=red;bg=gray;options=bold>-d</> - The value is used to identify a development package.
    <fg=red;bg=gray;options=bold>--production</>, <fg=red;bg=gray;options=bold>-p</>  - The value is used to identify a production package.
HELP;

        $this
            ->setDescription('Prints the build specification data.')
            ->setHelp($help)
            ->addOption('buildName', 'b', InputOption::VALUE_REQUIRED, 'The build specification (the list of modules and default data). The build configurations are stored as yaml files in "service-tool/resources". Specify the "service-tool/resources" related file path with no extension., e.g., "build/default"', 'build/default')
            ->addOption('language', 'l', InputOption::VALUE_REQUIRED, 'The build language. The value is used for a localized package name.', 'en')
            ->addOption('development', 'd', InputOption::VALUE_NONE, 'The value is used to identify a development package.')
            ->addOption('production', 'p', InputOption::VALUE_NONE, 'The value is used to identify a production package.');
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $buildName  = $input->getOption('buildName');
        $lang       = $input->getOption('language');

        $output->writeln('<info>Installation command:</info>');
        $output->write("\t<comment>./install.sh -b {$buildName} -l {$lang}");

        $additional = 'additional/dev';
        if ($input->getOption('development')) {
            $additional = 'additional/dev';
            $output->write(' -d');
        } elseif ($input->getOption('production')) {
            $additional = 'additional/prod';
            $output->write(' -p');
        }

        $output->writeln('</comment>');

        $configurationFile = new ConfigurationFile(__DIR__ . "/../../resources/{$buildName}.yaml", ['Lang' => $lang]);
        if ($additional) {
            $configurationFile->setAdditionalMutationFiles([__DIR__ . "/../../resources/{$additional}.yaml"]);
        }

        foreach ($configurationFile->getConfiguration() as $field => $data) {
            $output->writeln("<info>$field :</info>");
            foreach ($data as $item) {
                $output->writeln("\t<comment>$item</comment>");
            }
        }

        return Command::SUCCESS;
    }
}
