<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Command;

use App\Deployment\Runner;
use App\Domain\ModuleDomain;
use App\Entity\Scenario;
use App\Exception\CheckPermissionsException;
use App\Exception\CheckUpgradeRequirementsException;
use App\Exception\ClearServiceToolCacheException;
use App\Exception\DownloadPackException;
use App\Exception\GetHashException;
use App\Exception\GetUpgradesException;
use App\Exception\RebuildIsAlreadyStartedException;
use App\Exception\UpgradeException;
use App\Marketplace\Marketplace;
use App\Operation\Build\Upgrade\CheckUpgradeRequirements;
use App\Operation\Build\Upgrade\DownloadPacks;
use App\Operation\Build\RemoveActiveScenario;
use App\Operation\Build\Upgrade\GenerateUpgradeEntries;
use App\Operation\Build\Upgrade\CheckPermissions;
use App\Operation\Build\Upgrade\GenerateTransitions;
use App\Operation\Build\RebuildFlag;
use App\Output\XCartConsoleOutput;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

final class UpgradeCommand extends Command
{
    /**
     * @var string
     */
    protected static $defaultName = 'xcst:upgrade';

    private Runner $runner;

    private XCartConsoleOutput $output;

    private GenerateTransitions $generateTransitions;

    private GenerateUpgradeEntries $generateUpgradeEntries;

    private CheckPermissions $checkPermissions;

    private EntityManagerInterface $entityManager;

    private RebuildFlag $rebuildFlag;

    private Marketplace $marketplace;

    private ModuleDomain $moduleDomain;

    private RemoveActiveScenario $removeActiveScenario;

    private DownloadPacks $downloadPacks;

    private CheckUpgradeRequirements $checkUpgradeRequirements;

    public function __construct(
        Runner                   $runner,
        XCartConsoleOutput       $output,
        GenerateTransitions      $generateTransitions,
        GenerateUpgradeEntries   $generateUpgradeEntries,
        CheckPermissions         $checkPermissions,
        EntityManagerInterface   $entityManager,
        RebuildFlag              $rebuildFlag,
        Marketplace              $marketplace,
        ModuleDomain             $moduleDomain,
        RemoveActiveScenario     $removeActiveScenario,
        DownloadPacks            $downloadPacks,
        CheckUpgradeRequirements $checkUpgradeRequirements
    ) {
        parent::__construct();

        $this->runner                   = $runner;
        $this->output                   = $output;
        $this->generateTransitions      = $generateTransitions;
        $this->generateUpgradeEntries   = $generateUpgradeEntries;
        $this->checkPermissions         = $checkPermissions;
        $this->entityManager            = $entityManager;
        $this->rebuildFlag              = $rebuildFlag;
        $this->marketplace              = $marketplace;
        $this->moduleDomain             = $moduleDomain;
        $this->removeActiveScenario     = $removeActiveScenario;
        $this->downloadPacks            = $downloadPacks;
        $this->checkUpgradeRequirements = $checkUpgradeRequirements;
    }

    protected function configure()
    {
        $this
            ->setDescription('Upgrade')
            ->addOption(
                'force',
                'f',
                InputOption::VALUE_NONE,
                'Force rebuild (Remove active scenario if exists)'
            )
            ->addOption(
                'type',
                't',
                InputOption::VALUE_REQUIRED,
                'Upgrade Type (Possible values: build, minor, major)',
                ''
            )
            ->addOption(
                'list',
                'l',
                InputOption::VALUE_REQUIRED,
                'Modules list to upgrade (example: -lCDev-GoSocial,XC-News)',
                ''
            )
            ->addOption(
                'localList',
                'm',
                InputOption::VALUE_OPTIONAL,
                'Manually loaded module list to upgrade (example: -mCDev-GoSocial,XC-News)',
                ''
            )
            ->addOption(
                'packList',
                'p',
                InputOption::VALUE_OPTIONAL,
                'Loaded module packs list to upgrade (example: -pXC-News:5.5.1.2) - (var/packs/XC-News-v5.5.1.2.tar.gz)',
                ''
            );
    }

    /**
     * @return bool whether the command should be publicly shown or not
     */
    public function isHidden()
    {
        return true;
    }

    /**
     * @throws GetHashException
     * @throws GetUpgradesException
     * @throws RebuildIsAlreadyStartedException
     * @throws UpgradeException
     * @throws CheckPermissionsException
     * @throws DownloadPackException
     * @throws CheckUpgradeRequirementsException
     * @throws \JsonException
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        if ($input->getOption('force')) {
            ($this->removeActiveScenario)();
        } else {
            $this->rebuildFlag->check();
        }

        $this->output->setOutput($output);

        $localList = $input->hasParameterOption(['--localList', '-m']) ? $this->parseModulesList($input->getOption('localList')) : null;
        $packList = $input->hasParameterOption(['--packList', '-p']) ? $this->parseModulesList($input->getOption('packList')) : null;

        if ($localList) {
            $modulesToUpgrade = $this->getLoadedModulesToUpgrade($localList);
            $upgradeEntries   = [];
            ($this->checkUpgradeRequirements)($modulesToUpgrade);

        } elseif ($packList) {
            $modulesToUpgrade = $this->getLoadedModulesToUpgrade($packList, true);
            ($this->checkUpgradeRequirements)($modulesToUpgrade);
            $upgradeEntries = ($this->generateUpgradeEntries)($packList, true);
            ($this->checkPermissions)($upgradeEntries);

        } else {
            $type = $input->getOption('type');
            if (!in_array($type, Marketplace::UPGRADE_TYPES, true)) {
                throw UpgradeException::fromWrongUpgradeType();
            }

            $availableUpgrades = $this->marketplace->getUpgrades()[$type] ?? [];

            if (!$availableUpgrades) {
                throw UpgradeException::fromNoUpgradesForType($type);
            }

            $upgradeList      = $this->parseModulesList($input->getOption('list'));
            $modulesToUpgrade = $this->getModulesToUpgrade($availableUpgrades, $upgradeList);

            if (!$modulesToUpgrade) {
                throw UpgradeException::fromNoUpgrades();
            }

            ($this->checkUpgradeRequirements)($modulesToUpgrade);
            $upgradeEntries   = ($this->generateUpgradeEntries)($modulesToUpgrade);

            ($this->checkPermissions)($upgradeEntries);
            ($this->downloadPacks)($modulesToUpgrade);
        }

        $scenario = new Scenario();
        $scenario->setType(Scenario::TYPE_UPGRADE);
        $scenario->setTransitions(
            ($this->generateTransitions)($modulesToUpgrade)
        );
        $scenario->setMetaData(['upgradeEntries' => $upgradeEntries]);

        $this->entityManager->persist($scenario);

        try {
            ($this->runner)($scenario, $this->output);
        } catch (ClearServiceToolCacheException $e) {
            // After move_packs step a new runtime should start with a new up-to-date cache
            // "exit(0)" is used here because there will be some problems with cache if "return Command::SUCCESS" will be used (see ECOM-2078)
            exit(0);
        }

        return Command::FAILURE;
    }

    private function parseModulesList($list): array
    {
        return array_filter(explode(',', $list));
    }

    private function getModulesToUpgrade(array $upgrades, array $upgradeList): array
    {
        if ($upgradeList) {
            $result = array_filter(
                $upgrades,
                static fn($moduleId) => in_array($moduleId, $upgradeList, true),
                ARRAY_FILTER_USE_KEY
            );
        } else {
            $result = $upgrades;
        }

        return $result;
    }

    private function getLoadedModulesToUpgrade(array $list, $isPack = false): array
    {
        $modulesToUpgrade = [];
        foreach ($list as $module) {
            if ($isPack) {
                [$moduleId, $version] = explode(':', $module);
            } else {
                $moduleData = $this->moduleDomain->readModuleInfo($module);
                [$moduleId, $version] = [$module, $moduleData['version']];
            }

            $modulesToUpgrade[$moduleId] = ['version' => MarketPlace::explodeVersion($version)];
        }

        return $modulesToUpgrade;
    }
}
