<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Command;

use App\Deployment\Runner;
use App\Domain\ModuleDomain;
use App\Entity\Scenario;
use App\Exception\BuildException;
use App\Exception\CheckPermissionsException;
use App\Exception\ClearServiceToolCacheException;
use App\Exception\DownloadModulesException;
use App\Exception\DownloadPackException;
use App\Exception\RebuildIsAlreadyStartedException;
use App\Exception\RemoveModulesException;
use App\Operation\Build\CheckRemovePermissions;
use App\Operation\Build\GenerateTransitions;
use App\Operation\Build\InstallModules\CheckPermissions;
use App\Operation\Build\InstallModules\DownloadPacks;
use App\Operation\Build\RebuildFlag;
use App\Operation\Build\RemoveActiveScenario;
use App\Output\XCartConsoleOutput;
use App\Repository\ModuleRepository;
use Doctrine\ORM\EntityManagerInterface;
use JsonException;
use MJS\TopSort\CircularDependencyException;
use MJS\TopSort\ElementNotFoundException;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

final class RebuildCommand extends Command
{
    /**
     * @var string
     */
    protected static $defaultName = 'xcst:rebuild';

    private Runner $runner;

    private XCartConsoleOutput $output;

    private GenerateTransitions $generateTransitions;

    private DownloadPacks $downloadPacks;

    private CheckPermissions $checkPermissions;

    private EntityManagerInterface $entityManager;

    private RebuildFlag $rebuildFlag;

    private RemoveActiveScenario $removeActiveScenario;

    private CheckRemovePermissions $checkRemovePermissions;

    private ModuleDomain $moduleDomain;

    private ModuleRepository $moduleRepository;

    public function __construct(
        Runner                 $runner,
        XCartConsoleOutput     $output,
        GenerateTransitions    $generateTransitions,
        DownloadPacks          $downloadPacks,
        CheckPermissions       $checkPermissions,
        EntityManagerInterface $entityManager,
        RebuildFlag            $rebuildFlag,
        RemoveActiveScenario   $removeActiveScenario,
        CheckRemovePermissions $checkRemovePermissions,
        ModuleDomain           $moduleDomain,
        ModuleRepository       $moduleRepository
    ) {
        parent::__construct();

        $this->runner                 = $runner;
        $this->output                 = $output;
        $this->generateTransitions    = $generateTransitions;
        $this->downloadPacks          = $downloadPacks;
        $this->checkPermissions       = $checkPermissions;
        $this->entityManager          = $entityManager;
        $this->rebuildFlag            = $rebuildFlag;
        $this->removeActiveScenario   = $removeActiveScenario;
        $this->checkRemovePermissions = $checkRemovePermissions;
        $this->moduleDomain           = $moduleDomain;
        $this->moduleRepository       = $moduleRepository;
    }

    protected function configure()
    {
        $help = <<< HELP
Rebuilds code and cache. It also can be used to install/uninstall and enable/disable modules. 

<info>Options:</info>
    <fg=red;bg=gray;options=bold>--force</>, <fg=red;bg=gray;options=bold>-f</> - Deletes the current build data and starts a new build creation in case of a failure.
    <fg=red;bg=gray;options=bold>--enable</>    - A comma-separated list of modules due to enabling. Modules are listed in the "{AuthorId}-{ModuleId}" format.
    <fg=red;bg=gray;options=bold>--disable</>   - A comma-separated list of modules due to disabling. Modules are listed in the "{AuthorId}-{ModuleId}" format.
    <fg=red;bg=gray;options=bold>--install</>   - A comma-separated list of modules due to installation. Modules are listed in the "{AuthorId}-{ModuleId}". Some modules may require a registered license before installation.
    <fg=red;bg=gray;options=bold>--remove</>    - A comma-separated list of modules due to uninstall. Modules are listed in the "{AuthorId}-{ModuleId}".

<info>For example:</info>
    <fg=red;bg=gray;options=bold>./bin/service xcst:rebuild --force</>
    <fg=red;bg=gray;options=bold>./bin/service xcst:rebuild --enable=CDev-Catalog,CDev-Egoods</>
HELP;

        $this
            ->setDescription('Rebuilds code and cache. It also can be used to install/uninstall and enable/disable modules.')
            ->setHelp($help)
            ->addOption(
                'force',
                'f',
                InputOption::VALUE_NONE,
                'Deletes the current build data and starts a new build creation in case of a failure.'
            )
            ->addOption(
                'enable',
                null,
                InputOption::VALUE_REQUIRED,
                'A comma-separated list of modules due to enabling. Modules are listed in the "{AuthorId}-{ModuleId}" format.',
                ''
            )
            ->addOption(
                'disable',
                null,
                InputOption::VALUE_REQUIRED,
                'A comma-separated list of modules due to disabling. Modules are listed in the "{AuthorId}-{ModuleId}" format.',
                ''
            )
            ->addOption(
                'install',
                null,
                InputOption::VALUE_REQUIRED,
                'A comma-separated list of modules due to installation. Modules are listed in the "{AuthorId}-{ModuleId}". Some modules may require a registered license before installation.',
                ''
            )
            ->addOption(
                'remove',
                null,
                InputOption::VALUE_REQUIRED,
                'A comma-separated list of modules due to uninstall. Modules are listed in the "{AuthorId}-{ModuleId}',
                ''
            );
    }

    /**
     * @throws BuildException
     * @throws CheckPermissionsException
     * @throws CircularDependencyException
     * @throws ElementNotFoundException
     * @throws JsonException
     * @throws RebuildIsAlreadyStartedException
     * @throws DownloadModulesException
     * @throws DownloadPackException
     * @throws ClearServiceToolCacheException
     * @throws RemoveModulesException
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        if ($input->getOption('force')) {
            ($this->removeActiveScenario)();
        } else {
            $this->rebuildFlag->check();
        }

        $this->output->setOutput($output);

        $modulesToEnable  = $this->parseModulesList($input->getOption('enable'));
        $modulesToDisable = $this->parseModulesList($input->getOption('disable'));
        $modulesToInstall = $this->parseModulesList($input->getOption('install'));
        $modulesToRemove  = $this->parseModulesList($input->getOption('remove'));

        if (array_intersect($modulesToDisable, $modulesToEnable, $modulesToInstall)) {
            $this->output->addErrorMessage('The list of modules to disable and the list of modules to install/enable have the same module ids. Fix this and retry the command.');

            return Command::FAILURE;
        }

        $scenario = new Scenario();
        $scenario->setType(Scenario::TYPE_REBUILD);

        if ($modulesToInstall) {
            $packPaths = ($this->downloadPacks)($modulesToInstall);
            ($this->checkPermissions)($modulesToInstall);


            // we have to add the module to $this->moduleRepository in order to form proper dependency map. XCB-1512
            $modulesToCleanUpOnError = array_keys($packPaths);
            foreach ($packPaths as $moduleId => $path) {
                $bkpSourcePath = $this->moduleDomain->getSourcePath();
                $this->moduleDomain->setSourcePath($path);
                $moduleInfo = $this->moduleDomain->readModuleInfo($moduleId);
                $module     = $this->moduleRepository->createModuleFromArray($moduleInfo);
                $this->entityManager->persist($module);
                $this->moduleDomain->setSourcePath($bkpSourcePath);
            }

            $this->entityManager->flush();

            $scenario->setMetaData(['packPaths' => $packPaths]);
        }

        if ($modulesToRemove) {
            ($this->checkRemovePermissions)($modulesToRemove);
        }

        try {
            $scenario->setTransitions(
                ($this->generateTransitions)(
                    $modulesToEnable,
                    $modulesToDisable,
                    $modulesToRemove,
                    $modulesToInstall
                )
            );
        } catch (BuildException|JsonException|CircularDependencyException|ElementNotFoundException $e) {
            if (!empty($modulesToCleanUpOnError)) {
                $this->moduleRepository->removeLocalModules($modulesToCleanUpOnError);
            }
            throw $e;
        }

        $this->entityManager->persist($scenario);

        ($this->runner)($scenario, $this->output);

        return Command::SUCCESS;
    }

    private function parseModulesList(string $list): array
    {
        return array_filter(explode(',', $list));
    }
}
