<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Command;

use App\Domain\ModuleDomain;
use App\Domain\XCart;
use App\Operation\Pack\PackerConfig\PackerConfig;
use App\Operation\Pack\PackerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

final class PackModuleCommand extends Command
{
    protected static $defaultName = 'xcst:pack-module';

    private XCart $XCart;

    private ModuleDomain $moduleDomain;

    private PackerInterface $gitPacker;

    private PackerInterface $localPacker;

    public function __construct(
        XCart $XCart,
        ModuleDomain $moduleDomain,
        PackerInterface $gitPacker,
        PackerInterface $localPacker
    ) {
        parent::__construct();

        $this->XCart        = $XCart;
        $this->moduleDomain = $moduleDomain;
        $this->gitPacker    = $gitPacker;
        $this->localPacker  = $localPacker;
    }

    protected function configure()
    {
        $help = <<< HELP
Creates a module package that can be uploaded to the Marketplace or installed in a store. 

<info>Options:</info>
    <fg=red;bg=gray;options=bold>--source</>, <fg=red;bg=gray;options=bold>-s</>      - The code source: git or local. If local is selected, the package is compiled from the current code. If git is selected, the package is compiled from the current git branch.
    <fg=red;bg=gray;options=bold>--modules</>, <fg=red;bg=gray;options=bold>-m</>     - A comma-separated list of modules for the package. Each module is a separate package. Modules are listed in the "{AuthorId}-{ModuleId}" format.
    <fg=red;bg=gray;options=bold>--all-modules</>, <fg=red;bg=gray;options=bold>-a</> - Creates all modules package.

<info>For example:</info>
    <fg=red;bg=gray;options=bold>./bin/service xcst:pack-module --source=git --modules=CDev-Catalog,CDev-Egoods</>
    <fg=red;bg=gray;options=bold>./bin/service xcst:pack-module --source=git --all-modules</>
HELP;

        $this
            ->setDescription('Creates a module package.')
            ->setHelp($help)
            ->addOption('source', 's', InputOption::VALUE_REQUIRED, 'The code source: git or local. If local is selected, the package is compiled from the current code. If git is selected, the package is compiled from the current git branch.', 'git')
            ->addOption('modules', 'm', InputOption::VALUE_REQUIRED, 'A comma-separated list of modules for the package. Each module is a separate package. Modules are listed in the "{AuthorId}-{ModuleId}" format.')
            ->addOption('all-modules', 'a', InputOption::VALUE_NONE, 'Creates all modules package.');
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $source = $input->getOption('source');

        if ($source === 'local') {
            $packer = $this->localPacker;
        } elseif ($source === 'git') {
            $packer = $this->gitPacker;
        } else {
            $output->writeln("<error>Wrong source given: {$source}</error>");

            return Command::FAILURE;
        }

        $sourcePath = $this->XCart->getSourcePath();
        $this->moduleDomain->setSourcePath($sourcePath);

        $modules = $input->getOption('modules');
        $modules = $this->getModules($modules ? array_filter(explode(',', $modules)) : [], $input->getOption('all-modules'));

        if (!$modules) {
            $output->writeln("<error>Empty modules list to pack</error>");

            return Command::FAILURE;
        }

        $config = new PackerConfig();
        $config->setSourcePath($sourcePath)
            ->setOutputPath($sourcePath . 'var/packs/')
            ->setModules($modules);

        $output->writeln('<info>Packing modules</info>');
        $output->writeln("<info>Source:</info> <comment>{$source}</comment>");
        $output->writeln('<info>Modules:</info> <comment>' . implode(', ', $modules) . '</comment>');

        $result = $packer->pack($config);

        $output->writeln('<info>Modules are successfully packed: </info>');

        foreach ($result as $path) {
            $output->writeln("\t<comment>{$path}</comment>");
        }

        return Command::SUCCESS;
    }

    private function getModules(array $modules, bool $allModules = false): array
    {
        $allModuleIds = $this->moduleDomain->searchLocalModules();

        if ($allModules) {
            return $allModuleIds;
        }

        return array_filter($modules, static fn ($moduleId) => in_array($moduleId, $allModuleIds, true));
    }
}
