<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Command;

use App\Domain\ModuleDomain;
use App\Domain\XCart;
use App\Entity\ConfigurationFile;
use App\Operation\Pack\PackerConfig\PackerConfig;
use App\Operation\Pack\PackerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

final class PackDistrCommand extends Command
{
    protected static $defaultName = 'xcst:pack-distr';

    private XCart $XCart;

    private ModuleDomain $moduleDomain;

    private PackerInterface $gitPacker;

    private PackerInterface $localPacker;

    public function __construct(
        XCart $XCart,
        ModuleDomain $moduleDomain,
        PackerInterface $gitPacker,
        PackerInterface $localPacker
    ) {
        parent::__construct();

        $this->XCart        = $XCart;
        $this->moduleDomain = $moduleDomain;
        $this->gitPacker    = $gitPacker;
        $this->localPacker  = $localPacker;
    }

    protected function configure()
    {
        $help = <<< HELP
Creates an installation package. 

<info>Options:</info>
    <fg=red;bg=gray;options=bold>--source</>, <fg=red;bg=gray;options=bold>-s</>       - The code source: git or local. If local is selected, the package is compiled from the current code. If git is selected, the package is compiled from the current git branch.
    <fg=red;bg=gray;options=bold>--core-version</>, <fg=red;bg=gray;options=bold>-c</> - The core version. The value is used for the metadata and package name generation.
    <fg=red;bg=gray;options=bold>--build-name</>, <fg=red;bg=gray;options=bold>-b</>   - The build specification (the list of modules and default data). The build configurations are stored as yaml files in "service-tool/resources". Specify the "service-tool/resources" related file path with no extension., e.g., "build/default"
    <fg=red;bg=gray;options=bold>--lang</>, <fg=red;bg=gray;options=bold>-l</>         - The build language. The value is used for a localized package name.
    <fg=red;bg=gray;options=bold>--license-key</>, <fg=red;bg=gray;options=bold>-k</>  - A license key for module installation.
    <fg=red;bg=gray;options=bold>--dev-mode</>, <fg=red;bg=gray;options=bold>-d</>     - The value is used to identify a development package.
    <fg=red;bg=gray;options=bold>--prod-mode</>, <fg=red;bg=gray;options=bold>-p</>    - The value is used to identify a production package.

<info>For example:</info>
    <fg=red;bg=gray;options=bold>./bin/service xcst:pack-distr --source=git --core-version=5.5.0.1 --build-name=build/auto --lang=en --license-key=some-valid-license -p</>
HELP;

        $this
            ->setDescription('Creates an installation package.')
            ->setHelp($help)
            ->addOption('source', 's', InputOption::VALUE_REQUIRED, 'The code source: git or local. If local is selected, the package is compiled from the current code. If git is selected, the package is compiled from the current git branch.', 'git')
            ->addOption('core-version', 'c', InputOption::VALUE_REQUIRED, 'The core version. The value is used for the metadata and package name generation.')
            ->addOption('build-name', 'b', InputOption::VALUE_REQUIRED, 'The build specification (the list of modules and default data). The build configurations are stored as yaml files in "service-tool/resources". Specify the "service-tool/resources" related file path with no extension., e.g., "build/default"', 'build/default')
            ->addOption('lang', 'l', InputOption::VALUE_REQUIRED, 'The build language. The value is used for a localized package name.', 'en')
            ->addOption('license-key', 'k', InputOption::VALUE_REQUIRED, 'A license key for module installation.', '')
            ->addOption('dev-mode', 'd', InputOption::VALUE_NONE, 'The value is used to identify a development package.')
            ->addOption('prod-mode', 'p', InputOption::VALUE_NONE, 'The value is used to identify a production package.');
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $source = $input->getOption('source');

        if ($source === 'local') {
            $packer = $this->localPacker;
        } elseif ($source === 'git') {
            $packer = $this->gitPacker;
        } else {
            $output->writeln("<error>Wrong source given: {$source}</error>");

            return Command::FAILURE;
        }

        $buildName  = $input->getOption('build-name');
        $lang       = $input->getOption('lang');
        $dev        = $input->getOption('dev-mode');
        $prod       = $input->getOption('prod-mode');
        $licenseKey = $input->getOption('license-key');

        $configurationFile = new ConfigurationFile(__DIR__ . "/../../resources/{$buildName}.yaml", ['Lang' => $lang]);
        if ($prod) {
            $configurationFile->setAdditionalMutationFiles([__DIR__ . "/../../resources/additional/prod.yaml"]);
        } else {
            $configurationFile->setAdditionalMutationFiles([__DIR__ . "/../../resources/additional/dev.yaml"]);
        }

        $buildSpec       = "{$buildName};{$lang};" . ($prod ? 'prod' : 'dev');
        $configuration   = ['BuildSpec' => $buildSpec] + $configurationFile->getConfiguration();
        $enabledModules  = $configuration['EnabledModules'];
        $disabledModules = $configuration['DisabledModules'];
        $externalModules = $configuration['ExternalModules'];

        $version = $input->getOption('core-version');
        if (empty($version) || !preg_match('/^5\.\d+\.\d+\.\d+$/', $version)) {
            $output->writeln("<error>Wrong version given: {$version}</error>");

            return Command::FAILURE;
        }

        $config = new PackerConfig();
        $config->setSourcePath($sourcePath = $this->XCart->getSourcePath())
            ->setOutputPath($sourcePath . 'var/packs/')
            ->setName(basename($buildName))
            ->setVersion($version)
            ->setLanguage($lang)
            ->setModules($enabledModules)
            ->setExternalModules($externalModules)
            ->setDisabledModules($disabledModules)
            ->setLicenseKey($licenseKey)
            ->setInstallConfiguration($configuration);

        $output->writeln('<info>Packing distr</info>');
        $output->writeln("<info>Source:</info> <comment>{$source}</comment>");
        $output->writeln("<info>Version:</info> <comment>{$version}</comment>");
        $output->writeln("<info>Spec:</info> <comment>{$buildSpec}</comment>");
        $output->writeln('<info>Enabled modules:</info> <comment>' . implode(', ', $enabledModules) . '</comment>');

        if ($disabledModules) {
            $output->writeln('<info>Disabled modules:</info> <comment>' . implode(', ', $disabledModules) . '</comment>');
        }

        if ($externalModules) {
            $output->writeln('<info>External modules:</info> <comment>' . implode(', ', $externalModules) . '</comment>');
        }

        $result = $packer->pack($config);

        $output->writeln('<info>Distr is successfully packed: </info>');

        foreach ($result as $path) {
            $output->writeln("\t<comment>{$path}</comment>");
        }

        return Command::SUCCESS;
    }
}
