<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Command;

use App\Domain\XCart;
use App\Operation\ExecuteShellCommand;
use App\Operation\Pack\PackerConfig\PackerConfig;
use App\Operation\Pack\PackerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

final class PackCoreCommand extends Command
{
    protected static $defaultName = 'xcst:pack-core';

    private PackerInterface $gitPacker;

    private PackerInterface $localPacker;

    private XCart $XCart;

    public function __construct(
        XCart $XCart,
        PackerInterface $gitPacker,
        PackerInterface $localPacker
    ) {
        parent::__construct();

        $this->XCart       = $XCart;
        $this->gitPacker   = $gitPacker;
        $this->localPacker = $localPacker;
    }

    protected function configure()
    {
        $help = <<< HELP
Creates a core package due to update. 

<info>Options:</info>
    <fg=red;bg=gray;options=bold>--source</>, <fg=red;bg=gray;options=bold>-s</>       - The code source, supports git or local. If local is selected, the package will be created from the current code. If git is selected, the package will be created from the current git branch.
    <fg=red;bg=gray;options=bold>--core-version</>, <fg=red;bg=gray;options=bold>-c</> - The core version. The value is used for the metadata and package name generation, e.g., 5.5.0.0

<info>For example:</info>
    <fg=red;bg=gray;options=bold>./bin/service xcst:pack-core --core-version=5.5.0.1 --source=local</>
HELP;

        $this
            ->setDescription('Creates a core package.')
            ->setHelp($help)
            ->addOption('source', 's', InputOption::VALUE_REQUIRED, 'The code source, supports git or local. If local is selected, the package will be created from the current code. If git is selected, the package will be created from the current git branch.', 'git')
            ->addOption('core-version', 'c', InputOption::VALUE_REQUIRED, 'The core version. The value is used for the metadata and package name generation, e.g., 5.5.0.0');
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $source = $input->getOption('source');

        if ($source === 'local') {
            $packer = $this->localPacker;
        } elseif ($source === 'git') {
            $packer = $this->gitPacker;
        } else {
            $output->writeln("<error>Wrong source given: {$source}</error>");

            return Command::FAILURE;
        }

        $version = $input->getOption('core-version');
        if (!preg_match('/^5\.\d+\.\d+\.\d+$/', $version)) {
            $output->writeln("<error>Wrong version given: {$version}</error>");

            return Command::FAILURE;
        }

        $config = new PackerConfig();
        $config
            ->setSource($source)
            ->setSourcePath($sourcePath = $this->XCart->getSourcePath())
            ->setOutputPath($sourcePath . 'var/packs/')
            ->setVersion($version);

        $output->writeln("<info>Packing core</info>");
        $output->writeln("<info>Source:</info> <comment>{$source}</comment>");
        $output->writeln("<info>Version:</info> <comment>{$version}</comment>");

        $result = $packer->pack($config);

        $output->writeln('<info>Core is successfully packed: </info>');

        foreach ($result as $path) {
            $output->writeln("\t<comment>{$path}</comment>");
        }

        return Command::SUCCESS;
    }
}
