<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Command\LowLevel;

use App\Domain\ModuleDomain;
use App\Domain\XCart;
use App\Entity\ConfigurationFile;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

final class LoadFixturesCommand extends Command
{
    /**
     * @var string
     */
    protected static $defaultName = 'xcst:ll:load-fixtures';

    private XCart $XCart;

    private ModuleDomain $moduleDomain;

    public function __construct(
        XCart $XCart,
        ModuleDomain $moduleDomain
    ) {
        parent::__construct();

        $this->XCart        = $XCart;
        $this->moduleDomain = $moduleDomain;
    }

    protected function configure()
    {
        $this
            ->setDescription('Load fixtures')
            ->addArgument('buildName', InputArgument::REQUIRED, 'Build name')
            ->addArgument('lang', InputArgument::REQUIRED, 'Language')
            ->addArgument('additional', InputArgument::OPTIONAL, 'Additional mutations');
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $buildName  = $input->getArgument('buildName');
        $lang       = $input->getArgument('lang');
        $additional = $input->getArgument('additional');

        $configurationFile = new ConfigurationFile(__DIR__ . "/../../../resources/{$buildName}.yaml", ['Lang' => $lang]);
        if ($additional) {
            $configurationFile->setAdditionalMutationFiles([__DIR__ . "/../../../resources/{$additional}.yaml"]);
        }

        $configuration = $configurationFile->getConfiguration();

        $output->write('Get file list to load');

        $files = [];

        $sourcePath = $this->XCart->getSourcePath();

        foreach ($configuration['Data'] as $file) {
            $files[] = $sourcePath . $file;
        }

        foreach ($configuration['EnabledModules'] as $module) {
            foreach ($this->moduleDomain->searchModuleInstallFixtures($module) as $fixture) {
                $files[] = (string) $fixture;
            }
        }

        foreach ($configuration['DemoData'] as $file) {
            $files[] = $sourcePath . $file;
        }

        $output->writeln(sprintf(' <info>%s files to load</info>', count($files)));

        $output->write('Load fixtures');

        $this->XCart->run('xcart:service:load-fixtures', [implode(',', $files)]);

        $output->writeln(' <info>OK</info>');

        return Command::SUCCESS;
    }
}
