<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Command\LowLevel;

use App\Domain\XCart;
use App\Logic\GetInstallRequirements;
use App\Logic\InstallRequirements\XCartInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Helper\Table;
use Symfony\Component\Console\Helper\TableCell;
use Symfony\Component\Console\Helper\TableCellStyle;
use Symfony\Component\Console\Helper\TableSeparator;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

final class InstallRequirementsCommand extends Command
{
    public const DESCRIPTION_COLUMN_WIDTH = 160;

    /**
     * @var string
     */
    protected static $defaultName = 'xcst:ll:install-requirements';

    private XCart $XCart;

    private GetInstallRequirements $requirements;

    private TranslatorInterface $translator;

    /**
     * @var int|mixed
     */
    private $thirdColumnWidth;

    public function __construct(
        XCart $XCart,
        GetInstallRequirements $requirements,
        TranslatorInterface $translator
    ) {
        parent::__construct();

        $this->XCart        = $XCart;
        $this->requirements = $requirements;
        $this->translator   = $translator;
    }

    protected function configure()
    {
        $this->setDescription('Check install requirements')
            ->addArgument('descriptionColumnWidth', InputArgument::OPTIONAL, 'The width of the third column. ' . static::DESCRIPTION_COLUMN_WIDTH . ' by default')
            ->addArgument('-v', InputArgument::OPTIONAL, 'Show all requirements anyway');
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $this->thirdColumnWidth = (int) $input->getArgument('descriptionColumnWidth') ?: static::DESCRIPTION_COLUMN_WIDTH;

        $result       = [];
        $requirements = $this->requirements->getResult();

        foreach ($requirements as $name => $requirement) {
            $requirement['status'] = $requirement['state'] === XCartInterface::STATE_SUCCESS;
            $requirement['title']  = $this->getRequirementTranslation($requirement['title']) ?: $requirement['title'];

            $requirement['error_description'] = $this->getRequirementTranslation($requirement['description'], $requirement['data']);
            $requirement['description']       = $this->getRequirementTranslation($requirement['label_message'], $requirement['data']);
            $requirement['kb_description']    = $this->getRequirementTranslation($requirement['kb_message'], $requirement['data']);

            if (
                !$requirement['status'] // there is an error
                || ($output->getVerbosity() > OutputInterface::VERBOSITY_NORMAL)
            ) {
                if (!$requirement['status']) {
                    if ($requirement['level'] === XCartInterface::LEVEL_CRITICAL) {
                        $criticalErrorIsPresent = true;
                    } else {
                        $nonCriticalErrorIsPresent = true;
                    }
                }
                $result[$name] = $requirement;
            }
        }

        $criticalResult = array_filter($result, static function ($item) {
            return $item['level'] === XCartInterface::LEVEL_CRITICAL;
        });

        $nonCriticalResult = array_filter($result, static function ($item) {
            return $item['level'] !== XCartInterface::LEVEL_CRITICAL;
        });

        if ($criticalResult) {
            $output->writeln('Critical requirements:');
            $this->showRequirementsTable($criticalResult, $output, !empty($criticalErrorIsPresent));
        }

        if ($nonCriticalResult) {
            if ($criticalResult) {
                $output->writeln('');
            }
            $output->writeln('Noncritical requirements:');
            $this->showRequirementsTable($nonCriticalResult, $output, !empty($nonCriticalErrorIsPresent), false);
        }

        if (!$nonCriticalResult && !$criticalResult) {
            $output->write('Check install requirements');
            $output->writeln(' <info>OK</info>');
        }

        return $criticalResult ? Command::FAILURE : Command::SUCCESS;
    }

    protected function showRequirementsTable(array $result, OutputInterface $output, $moveErrorUp = false, $isCritical = true): void
    {
        if ($moveErrorUp) {
            usort($result, static function ($a, $b) {
                return $a['status'] === $b['status'] ? 0 : 1;
            });
        }

        $table = new Table($output);
        $table->setColumnMaxWidth(2, $isCritical ? $this->thirdColumnWidth : $this->thirdColumnWidth - 6);

        $styleRed   = $isCritical ? new TableCellStyle(['fg' => 'red']) : new TableCellStyle(['cellFormat' => '<comment>%s</comment>']);
        $styleGreen = new TableCellStyle(['cellFormat' => '<info>%s</info>']);

        $i = 0;
        foreach ($result as $row) {
            $okMessage     = $row['status'] ? 'OK' : ($isCritical ? 'ERR' : 'WARN');
            $okMessageCell = new TableCell($okMessage, ['style' => $row['status'] ? $styleGreen : $styleRed]);
            if ($row['status']) {
                $table->addRow([$row['title'], $okMessageCell]);
            } else {
                $description = array_filter(array_unique([$row['error_description'], $row['description'], $row['kb_description']]));

                $table->addRow([$row['title'], $okMessageCell, implode("\n\n", $description)]);

                if ($i++ < count($result) - 1) {
                    $table->addRow(new TableSeparator());
                }
            }

        }
        $table->render();
    }

    private function getRequirementTranslation(string $name, array $data = [])
    {
        $arguments = array_filter($data, 'is_scalar') ?: [];
        $translated = $this->translator->trans($name, $arguments);

        return strip_tags($translated, '<a><href>');
    }
}
