<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Command\LowLevel;

use App\Domain\ModuleDomain;
use App\Entity\Module;
use App\Repository\ModuleRepository;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

final class InitStorageCommand extends Command
{
    /**
     * @var string
     */
    protected static $defaultName = 'xcst:ll:init-storage';

    private ModuleDomain $moduleDomain;

    private ModuleRepository $moduleRepository;

    public function __construct(
        ModuleDomain $moduleDomain,
        ModuleRepository $moduleRepository
    ) {
        parent::__construct();

        $this->moduleDomain      = $moduleDomain;
        $this->moduleRepository  = $moduleRepository;
    }

    protected function configure()
    {
        $this->setDescription('Init modules storage. (The information of local modules will be dropped and will be read from the source files)');
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $output->write('Read local modules');

        $modules = [];

        foreach ($this->moduleDomain->searchLocalModules() as $module) {
            $modules[] = $this->moduleDomain->readModuleInfo($module);
        }

        if ($modules) {
            $output->writeln(sprintf(' <info>[%s]</info>', count($modules)));

            $output->write('Save modules to database');

            $this->moduleRepository->deleteBySource(Module::SOURCE_LOCAL);
            $this->moduleRepository->createModules($modules);

            $output->writeln(' <info>[OK]</info>');

            return Command::SUCCESS;
        }

        $output->writeln('No modules found');

        return Command::FAILURE;
    }
}
