<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Command;

use App\Domain\XCart;
use App\Marketplace\Marketplace;
use App\Operation\Integrity\ActualFilesRetriever;
use App\Operation\Integrity\ComparedFileRetriever;
use App\Operation\Integrity\CoreHashRetriever;
use App\Operation\Integrity\GenerateViolations;
use App\Operation\Integrity\ModuleHashRetriever;
use Exception;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

final class IntegrityCheckCommand extends Command
{
    protected static $defaultName = 'xcst:integrity-check';

    private const MODE_ACTUAL = 'actual';

    private const MODE_COMPARE = 'compared';

    private XCart $XCart;

    private CoreHashRetriever $coreHashRetriever;

    private ModuleHashRetriever $moduleHashRetriever;

    private ActualFilesRetriever $actualFilesRetriever;

    private ComparedFileRetriever $comparedFileRetriever;

    private GenerateViolations $generateViolations;

    private Marketplace $marketplace;

    public function __construct(
        XCart                 $XCart,
        CoreHashRetriever     $coreHashRetriever,
        ModuleHashRetriever   $moduleHashRetriever,
        ActualFilesRetriever  $actualFilesRetriever,
        ComparedFileRetriever $comparedFileRetriever,
        GenerateViolations    $generateViolations,
        Marketplace           $marketplace
    ) {
        parent::__construct();

        $this->XCart                 = $XCart;
        $this->coreHashRetriever     = $coreHashRetriever;
        $this->moduleHashRetriever   = $moduleHashRetriever;
        $this->actualFilesRetriever  = $actualFilesRetriever;
        $this->comparedFileRetriever = $comparedFileRetriever;
        $this->generateViolations    = $generateViolations;
        $this->marketplace           = $marketplace;
    }

    protected function configure()
    {
        $this
            ->setDescription('Check the files for code modifications.')
            ->setHelp('Checks the files for code modifications. The modified files are substituted with the default ones during the core and modules update. Sometimes it is necessary to check the original files for possible modifications to prevent the custom patches loss. The file modifications are not recommended and can be done only as an exception. Options and arguments are not supported.')
            ->addOption('mode', 'm', InputOption::VALUE_REQUIRED, 'Compare file mode (actual or compared)', self::MODE_ACTUAL);
    }

    /**
     * @throws Exception
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);

        $coreKnown = ($this->coreHashRetriever)(
            MarketPlace::explodeVersion($this->XCart->getCoreVersion())
        );

        if (!isset($coreKnown['error'])) {
            $mode = $input->getOption('mode');

            if ($mode === self::MODE_ACTUAL) {
                $coreActual = ($this->actualFilesRetriever)($this->XCart->getSourcePath());
            } elseif ($mode === self::MODE_COMPARE) {
                $coreActual = ($this->comparedFileRetriever)($this->XCart->getSourcePath(), array_keys($coreKnown));
            } else {
                throw new \RuntimeException("Unknown mode: '{$mode}'");
            }

            $violations = ($this->generateViolations)($coreKnown, $coreActual);
            $this->printIntegrityViolations($io, Marketplace::CORE_MODULE_ID, $violations);
        } else {
            $io->writeln('<error>' . ucfirst(Marketplace::CORE_MODULE_ID) . ':</error>');
            $io->writeln($coreKnown['message']);
        }

        $moduleHash = ($this->moduleHashRetriever)(
            $this->marketplace->getInstalledModuleVersionHashes()
        );

        foreach ($moduleHash as $moduleId => $data) {
            $known = $data['result'] ?? null;
            if ($known) {
                if ($mode === self::MODE_ACTUAL) {
                    $actual = ($this->actualFilesRetriever)($data['path'], $moduleId);
                } elseif ($mode === self::MODE_COMPARE) {
                    $actual = ($this->comparedFileRetriever)($data['path'], array_keys($known), $moduleId);
                } else {
                    throw new \RuntimeException("Unknown mode: '{$mode}'");
                }

                $violations = ($this->generateViolations)($known, $actual);
                $this->printIntegrityViolations($io, $moduleId, $violations);
            } else {
                $io->writeln('<error>' . ucfirst($moduleId) . ':</error>');
                $io->writeln($data['error']);
            }
        }

        return Command::SUCCESS;
    }

    protected function printIntegrityViolations($io, $moduleId, $violations): void
    {
        if ($violations) {
            $this->moduleChangesOutput($io, $moduleId, $violations);
        } else {
            $io->writeln('<info>' . $moduleId . ' - OK' . '</info>');
        }
    }

    /**
     * @param SymfonyStyle $io
     * @param string       $moduleId
     * @param array        $moduleChanges
     *
     * @return void
     */
    protected function moduleChangesOutput(SymfonyStyle $io, string $moduleId, array $moduleChanges): void
    {
        $changesOutput = [];

        foreach ($moduleChanges as $change) {
            if ($change['type'] === 'added') {
                $changesOutput['added'][] = $change['filepath'];
            } elseif ($change['type'] === 'removed') {
                $changesOutput['removed'][] = $change['filepath'];
            } elseif ($change['type'] === 'modified') {
                $changesOutput['modified'][] = $change['filepath'];
            }
        }

        $io->title($moduleId);

        foreach ($changesOutput as $type => $values) {
            $this->printModuleOutput($io, $type, $changesOutput);
        }
    }

    protected function printModuleOutput(SymfonyStyle $io, string $type, array $changesOutput): void
    {
        $io->writeln('<fg=blue>' . ucfirst($type) . ':</>');
        $io->newLine();
        foreach ($changesOutput[$type] as $item) {
            $io->writeln($item);
        }

        $io->newLine();
    }
}
