<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Command;

use App\Deployment\Runner;
use App\Deployment\Step\CreateAdminProfile;
use App\Deployment\Step\ResizeImages;
use App\Domain\ModuleDomain;
use App\Entity\ConfigurationFile;
use App\Entity\Scenario;
use App\Exception\AdminCredentialsException;
use App\Exception\BuildException;
use App\Exception\ClearServiceToolCacheException;
use App\Operation\Build\GenerateTransitions;
use App\Operation\Build\ParseCredentials;
use App\Operation\Pack\GetCode\GetExternalModules;
use App\Output\XCartConsoleOutput;
use App\Repository\ModuleRepository;
use Doctrine\ORM\EntityManagerInterface;
use JsonException;
use MJS\TopSort\CircularDependencyException;
use MJS\TopSort\ElementNotFoundException;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

final class InstallCommand extends Command
{
    /**
     * @var string
     */
    protected static $defaultName = 'xcst:install';

    private Runner $runner;

    private XCartConsoleOutput $output;

    private GenerateTransitions $generateTransitions;

    private EntityManagerInterface $entityManager;

    private GetExternalModules $getExternalModules;

    private ModuleDomain $moduleDomain;

    private ModuleRepository $moduleRepository;

    private ParseCredentials $parseCredentials;

    public function __construct(
        Runner $runner,
        XCartConsoleOutput $output,
        GenerateTransitions $generateTransitions,
        EntityManagerInterface $entityManager,
        GetExternalModules $getExternalModules,
        ModuleDomain $moduleDomain,
        ModuleRepository $moduleRepository,
        ParseCredentials $parseCredentials
    ) {
        parent::__construct();

        $this->runner              = $runner;
        $this->output              = $output;
        $this->generateTransitions = $generateTransitions;
        $this->entityManager       = $entityManager;
        $this->getExternalModules  = $getExternalModules;
        $this->moduleDomain        = $moduleDomain;
        $this->moduleRepository    = $moduleRepository;
        $this->parseCredentials    = $parseCredentials;
    }

    protected function configure()
    {
        $this
            ->setDescription('Installation runner (do not use this command directly, use install.sh instead)')
            ->addArgument('buildName', InputArgument::REQUIRED, 'Build name')
            ->addArgument('lang', InputArgument::REQUIRED, 'Language')
            ->addArgument('additional', InputArgument::OPTIONAL, 'Additional mutations')
            ->addOption('adminCredentials', 'a', InputOption::VALUE_OPTIONAL, 'Admin login and password (Example: admin:password)', '')
            ->addOption('resizeAllImages', 'r', InputOption::VALUE_OPTIONAL, 'Resize all images (yes or no)', 'no')
            ->addOption('downloadExternalModules', 'm', InputOption::VALUE_OPTIONAL, 'Download external modules', 'no')
            ->addOption('licenseKey', 'k', InputOption::VALUE_OPTIONAL, 'License key', '');
    }

    /**
     * @throws AdminCredentialsException
     * @throws BuildException
     * @throws CircularDependencyException
     * @throws ClearServiceToolCacheException
     * @throws ElementNotFoundException
     * @throws JsonException
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $this->output->setOutput($output);

        $adminCredentials = $input->getOption('adminCredentials');
        $resizeAllImages = $input->getOption('resizeAllImages');
        $downloadExternalModules = $input->getOption('downloadExternalModules');
        $licenseKey = $input->getOption('licenseKey');

        $scenario = new Scenario();
        $scenario->setType(Scenario::TYPE_INSTALL);

        // Ignore input arguments if installation spec is present
        if (is_readable($installYaml = __DIR__ . '/../../../var/install.yaml')) {
            $configurationFile = new ConfigurationFile($installYaml);
        } else {
            $buildName  = $input->getArgument('buildName');
            $lang       = $input->getArgument('lang');
            $additional = $input->getArgument('additional');

            $configurationFile = new ConfigurationFile(__DIR__ . "/../../resources/{$buildName}.yaml", ['Lang' => $lang]);
            if ($additional) {
                $configurationFile->setAdditionalMutationFiles([__DIR__ . "/../../resources/{$additional}.yaml"]);
            }
        }

        $configuration = $configurationFile->getConfiguration();

        if ($downloadExternalModules === 'yes') {
            ($this->getExternalModules)(__DIR__ . "/../../../", $configuration['ExternalModules'], $licenseKey);

            $modules = [];
            foreach ($configuration['ExternalModules'] as $module) {
                $modules[] = $this->moduleDomain->readModuleInfo($module);
            }

            $this->moduleRepository->removeLocalModules($configuration['ExternalModules']);
            $this->moduleRepository->createModules($modules);
        }

        $modulesToEnabled = $configuration['EnabledModules'];
        if ($downloadExternalModules === 'yes' || is_readable($installYaml)) {
            $modulesToEnabled = array_merge($modulesToEnabled, $configuration['ExternalModules']);
        }

        $scenario->setTransitions(
            ($this->generateTransitions)($modulesToEnabled ?? [])
        );

        $scenario->setMetaData([
            'data'     => $configuration['Data'] ?? [],
            'demoData' => $configuration['DemoData'] ?? [],
        ]);

        $scenario->setStepData(
            CreateAdminProfile::class,
            $adminCredentials
                ? ($this->parseCredentials)($adminCredentials, $lang ?? 'en')
                : []
        );

        $scenario->setStepData(ResizeImages::class, ['resizeAllImages' => $resizeAllImages]);

        $this->entityManager->persist($scenario);

        ($this->runner)($scenario, $this->output);

        return Command::SUCCESS;
    }
}
