<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Command;

use App\Domain\XCart;
use App\Operation\Pack\CreatePack\CreateModulesPack;
use App\Operation\Pack\Finish\Finish;
use App\Operation\Pack\GetCode\GetTranslationModuleCode;
use App\Operation\Pack\Init\InitTranslationCode;
use App\Operation\Pack\PackerConfig\PackerConfig;
use JsonException;
use Psr\Log\LoggerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

final class GenerateTranslationModuleCommand extends Command
{
    use CommandHelpersTrait;

    protected const TEMPORARY_ARCHIVE_SUBFOLDER = InitTranslationCode::TEMPORARY_DATA_SUBFOLDER;

    protected const TEMPORARY_ARCHIVE_NAME = 'translation.tgz';

    /**
     * @var string
     */
    protected static $defaultName = 'xcst:generate-translation-module';

    protected function getCmdDescription(): string
    {
        return 'Generate translation module from CSV file.';
    }

    protected function getArgumentDescriptions(): array
    {
        return [
            'src' => 'CSV file path (relative to the X-Cart root folder).'
        ];
    }

    protected function getOptionDescriptions(): array
    {
        return [
            'lang'        => [
                'text'    => 'Language code.',
                'default' => 'language code from the first line in the CSV file',
            ],
            'author'      => [
                'text'    => 'Module author',
                'default' => 'XC',
            ],
            'name'        => [
                'text'    => 'Module name',
                'default' => '"{language code}Translation"',
            ],
            'boilerplate' => [
                'text'    => 'Path to the module boilerplate (the folder with all module-related files except config/install.yaml)',
                'default' => 'var/packs/boilerplate',
            ],
        ];
    }

    public function __construct(
        protected InitTranslationCode $initialActions,
        protected GetTranslationModuleCode $translationModuleCode,
        protected XCart $XCart,
        protected CreateModulesPack $packer,
        protected Finish $packFinisher,
        protected LoggerInterface $logger
    ) {
        parent::__construct();
    }

    protected function configure()
    {
        $this
            ->setDescription($this->getCmdDescription())
            ->setHelp(
                $this->generateHelp([
                    'arguments' => ['src'],
                    'options' => [
                        ['--lang', '-l'],
                        ['--author', '-a'],
                        '--name',
                        ['--boilerplate', '-bp']
                    ],
                    'examples' => [
                        'es.csv',
                        '--lang=es --boilerplate=var/tmp/boilerplate es-labels.csv',
                        '--author=XC --name=ESTranslation --lang=es es.csv',
                        '-lde de.csv'
                    ]
                ])
            )
            ->addOption(
                'lang',
                'l',
                InputOption::VALUE_REQUIRED,
                $this->getOptionDescription('lang')
            )
            ->addOption(
                'author',
                'a',
                InputOption::VALUE_REQUIRED,
                $this->getOptionDescription('author'),
                $this->getOptionDefaultValue('author')
            )
            ->addOption(
                'name',
                null,
                InputOption::VALUE_REQUIRED,
                $this->getOptionDescription('name')
            )
            ->addOption(
                'boilerplate',
                'bp',
                InputOption::VALUE_REQUIRED,
                $this->getOptionDescription('boilerplate'),
                $this->getOptionDefaultValue('boilerplate')
            )
            ->addArgument(
                'src',
                InputArgument::REQUIRED,
                $this->getOptionDescription('src')
            );
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $this->setInterfaces($input, $output);

        ($this->initialActions)(
            $this->XCart->getPacksPath()
        );

        $baseFilesData = $this->handleCommand(
            'xcart:service:generate-translation-base'
            . $this->appendOptions(['lang', 'author', 'name'])
            . ' ' . $input->getArgument('src'),
            false
        );

        if (is_string($baseFilesData)) {
            try {
                $data = json_decode($baseFilesData, true, 512, JSON_THROW_ON_ERROR);
            } catch (JsonException $e) {
                return $this->returnError(
                    'Error while decoding JSON answer from X-Cart: ' . $e->getMessage(),
                    [ 'Raw answer data' => $data ]
                );
            }

            $bpDir = $this->XCart->getSourcePath() . $input->getOption('boilerplate');

            if (!file_exists($bpDir) || !is_dir($bpDir)) {
                return $this->returnError("Cannot find boilerplate directory {$bpDir}");
            }

            if (!is_readable($bpDir)) {
                return $this->returnError("Cannot read from the boilerplate directory {$bpDir}");
            }

            $moduleDir = $data['dir'] ?? '';
            $lngModuleId = ($data['author'] ?? $this->getOptionDefaultValue('author')) . '-' . ($data['moduleName']);

            ($this->translationModuleCode)(
                $this->XCart->getSourcePath() . $input->getOption('boilerplate'),
                $moduleDir,
                $this->XCart->getPacksPath() . self::TEMPORARY_ARCHIVE_SUBFOLDER . self::TEMPORARY_ARCHIVE_NAME
            );

            $config = new PackerConfig();
            $config->setOutputPath(dirname($moduleDir, 4) . '/')
                ->setModules([$lngModuleId]);

            $modulePackPath = ($this->packer)($config)[0] ?? '';
            ($this->packFinisher)($config);

            $moduleLanguage = $data['lang'] ?? '';
            $entitiesAdded = $data['entities'] ?? 0;
            $entitiesIgnored = $data['ignored'] ?? 0;

            if ($moduleLanguage) {
                $output->writeln("Module language: <info>{$moduleLanguage}</info>");
            }

            if ($entitiesAdded > 0 && $modulePackPath) {
                $output->writeln("<options=bold>{$entitiesAdded}</> labels have been included into the module.");
                $output->writeln("Archive path: <info>{$modulePackPath}</info>");
            }

            if ($entitiesIgnored > 0 && !empty($data['ignoredPath'])) {
                $output->writeln("<options=bold>{$entitiesIgnored}</> labels ignored, full list is saved into <info>{$data['ignoredPath']}</info>");
            }
        } else {
            $this->returnError(
                'Incorrect answer from X-Cart',
                [
                    'answer' => $baseFilesData
                ]
            );
        }

        return Command::SUCCESS;
    }
}
