<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Command;

use App\Deployment\Runner;
use App\Entity\Scenario;
use App\Exception\ClearServiceToolCacheException;
use App\Output\XCartConsoleOutput;
use App\Repository\ScenarioRepository;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

final class ContinueCommand extends Command
{
    /**
     * @var string
     */
    protected static $defaultName = 'xcst:continue';

    private Runner $runner;

    private XCartConsoleOutput $output;

    private ScenarioRepository $scenarioRepository;

    public function __construct(
        Runner             $runner,
        XCartConsoleOutput $output,
        ScenarioRepository $scenarioRepository
    ) {
        parent::__construct();

        $this->runner             = $runner;
        $this->output             = $output;
        $this->scenarioRepository = $scenarioRepository;
    }

    protected function configure()
    {
        $this->setDescription('Continue last scenario');
    }

    /**
     * @throws ClearServiceToolCacheException
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $this->output->setOutput($output);

        $lastScenario = $this->scenarioRepository->findLastScenario();

        if (
            $lastScenario === null
            || $lastScenario->getState() === Scenario::STATE_DONE
        ) {
            $this->output->addErrorMessage('No unfinished last scenario');

            return Command::FAILURE;
        }

        $this->output
            ->addInitMessage('Skip already passed steps')
            ->addEndMessage('[OK]');

        ($this->runner)($lastScenario, $this->output);

        return Command::SUCCESS;
    }
}
