<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace App\Command;

use App\Exception\UpgradeException;
use App\Exception\GetUpgradesException;
use App\Marketplace\Marketplace;
use App\Marketplace\MarketplaceStorage;
use App\Repository\ModuleRepository;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

final class CheckForUpgradeCommand extends Command
{
    /**
     * @var string
     */
    protected static $defaultName = 'xcst:check-for-upgrade';

    private Marketplace $marketplace;

    private MarketplaceStorage $marketplaceStorage;

    private ModuleRepository $moduleRepository;

    public function __construct(
        Marketplace        $marketplace,
        MarketplaceStorage $marketplaceStorage,
        ModuleRepository   $moduleRepository
    ) {
        parent::__construct();

        $this->marketplace        = $marketplace;
        $this->moduleRepository   = $moduleRepository;
        $this->marketplaceStorage = $marketplaceStorage;
    }

    protected function configure()
    {
        $help = <<< HELP
Checks the core and modules update availability.
If the type is not specified, the number of modules (and the core) due for the update is printed for each type. If the type is specified, the list of modules due to update is printed with versions applicable to the selected type.

<info>Options:</info>
    <fg=red;bg=gray;options=bold>--type</>, <fg=red;bg=gray;options=bold>-t</> - The type of update (build, minor, major)
HELP;

        $this
            ->setDescription('Checks the core and modules update availability.')
            ->setHelp($help)
            ->addOption(
                'type',
                't',
                InputOption::VALUE_REQUIRED,
                'The type of update (build, minor, major)',
                ''
            );
    }

    /**
     * @throws UpgradeException
     * @throws GetUpgradesException
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $upgrades = $this->marketplace->getUpgrades();

        if ($upgradeType = $input->getOption('type')) {
            $this->validateUpgradeType($upgradeType);

            if (!isset($upgrades[$upgradeType])) {
                $output->writeln("<comment>No upgrades found for {$upgradeType} type</comment>");
            } else {
                foreach ($upgrades[$upgradeType] as $moduleId => $upgrade) {
                    $versionFrom = ($moduleId === Marketplace::CORE_MODULE_ID)
                        ? $this->marketplaceStorage->getValue('XCartCoreVersion')
                        : $this->moduleRepository->findByModuleId($moduleId)->getVersion();
                    $versionTo   = Marketplace::implodeVersion($upgrade['version']);

                    $output->writeln("{$versionFrom} => {$versionTo}\t<info>{$moduleId}</info>");
                }
            }
        } else {
            foreach ($upgrades as $type => $typeUpgrades) {
                $typeUpgradesCount = count($typeUpgrades);

                $output->writeln("{$type} <info>{$typeUpgradesCount}</info>");
            }
        }

        return Command::SUCCESS;
    }

    /**
     * @throws UpgradeException
     */
    private function validateUpgradeType(string $type): void
    {
        if (!in_array($type, Marketplace::UPGRADE_TYPES, true)) {
            throw UpgradeException::fromWrongUpgradeType();
        }
    }
}
