<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace App\Command;

use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use App\Domain\HookManagerDomain;

final class CallHookCommand extends Command
{
    protected static $defaultName = 'xcst:call-hook';

    private HookManagerDomain $hookManager;

    public function __construct(HookManagerDomain $hookManager)
    {
        parent::__construct();

        $this->hookManager = $hookManager;
    }

    protected function configure(): void
    {
        $help = <<< HELP
Executes lifetime hook for the service utility. Usually, hooks are executed automatically when a specific event takes place. However, sometimes manual hook execution may be required.
Service utility works only with the upgrade hooks. 

<info>Arguments:</info>
    <fg=red;bg=gray;options=bold>hookType</> - A hook type; possible values - upgrade

<info>Options:</info> For an upgrade type hook, specify the applicable version range. Do mind that hooks corresponding with the start version of the range will not be executed.
    <fg=red;bg=gray;options=bold>--versionFrom</> - The start version in the "x.x.x.x" format
    <fg=red;bg=gray;options=bold>--versionTo</>   - The end version in the "x.x.x.x" format

The service utility version corresponds with the core version as their release is done simultaneously. 

<info>For example:</info> Suppose there are hooks to update versions: "5.5.0.1", "5.5.0.2" и "5.5.0.3". You execute the following command:
    <fg=red;bg=gray;options=bold>./bin/service xcst:call-hook --versionForm=5.5.0.1 --versionTo=5.5.0.3 upgrade</>

The "5.5.0.1" hooks will not be executed, while the "5.5.0.2" and "5.5.0.3" hooks will.
HELP;

        $this
            ->setDescription('Executes service utility lifetime hook.')
            ->setHelp($help)
            ->addArgument('hookType', InputArgument::REQUIRED, 'A hook type; possible values - upgrade')
            ->addOption('versionFrom', null, InputOption::VALUE_REQUIRED, 'The start version in the "x.x.x.x" format')
            ->addOption('versionTo', null, InputOption::VALUE_REQUIRED, 'The end version in the "x.x.x.x" format');
    }

    /**
     * @throws \App\Exception\HookManagerException
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $count = $this->hookManager->runHook([
            'hookType'    => $input->getArgument('hookType'),
            'versionFrom' => $input->getOption('versionFrom') ?: '',
            'versionTo'   => $input->getOption('versionTo') ?: '',
        ]);

        $output->writeln("{$count} hook entities was called");

        $output->writeln('<info>OK</info>');

        return Command::SUCCESS;
    }
}
