#!/usr/bin/env php
<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

require_once dirname(__DIR__) . '/vendor/autoload.php';

/**
 * This script converts configuration form etc/config*.php ini files to .env.local and config/local/x_cart.yaml file
 */

if (in_array('-h', $argv, true)) {
    x_help();
}

const ENV_PREFIX = 'XCART_';

/**
 * section => [variable_name => variable_value]
 */
$configData = [];

/**
 * section => [variable_name => variable_value]
 */
$localYamlConfigData = [];

/**
 * ENV_VARIABLE_NAME => 'env_variable_value'
 */
$envData = [];

$XCRootPath  = x_get_absolute_path($argc > 1 ? $argv[1] : (__DIR__ . '/../../'));
$configPath  = $XCRootPath . '/etc';
$distEnvFile = $XCRootPath . '/.env';
$envFile     = $XCRootPath . '/.env.local';

$distYamlConfigFile = $XCRootPath . '/config/packages/x_cart.yaml';
$yamlConfigFile     = $XCRootPath . '/config/local/x_cart.yaml';

$configFiles = [
    'config.default.php',
    'config.dev.php',
    'config.php',
    'config.personal.php',
    'config.local.php',
];

$skipEnv = [
    'XCART_DATABASE_DETAILS_HOSTSPEC',
    'XCART_DATABASE_DETAILS_PORT',
    'XCART_DATABASE_DETAILS_DATABASE',
    'XCART_DATABASE_DETAILS_USERNAME',
    'XCART_DATABASE_DETAILS_PASSWORD',

    'XCART_IMAGES_DEFAULT_IMAGE',
    'XCART_IMAGES_DEFAULT_IMAGE_WIDTH',
    'XCART_IMAGES_DEFAULT_IMAGE_HEIGHT',
    'XCART_IMAGES_MAKE_PROGRESSIVE',
    'XCART_IMAGES_GENERATE_RETINA_IMAGES',

    'XCART_CACHE_TYPE',
    'XCART_CACHE_NAMESPACE',
    'XCART_CACHE_SERVERS',

    'XCART_LOG_DETAILS_TYPE',
    'XCART_LOG_DETAILS_NAME',
    'XCART_LOG_DETAILS_LEVEL',
    'XCART_LOG_DETAILS_IDENT',

    'XCART_SKIN_DETAILS_SKIN',
    'XCART_SKIN_DETAILS_LOCALE',

    'XCART_STOREFRONT_OPTIONS_AUTOCOMPLETE_STATES_FOR_COUNTRIES',

    'XCART_OTHER_TRUSTED_DOMAINS',
    'XCART_OTHER_TRUSTED_PROXIES',
    'XCART_OTHER_X_FRAME_OPTIONS',
    'XCART_OTHER_X_XSS_PROTECTION',
    'XCART_OTHER_CONTENT_SECURITY_POLICY',
    'XCART_OTHER_X_CONTENT_TYPE_OPTIONS',
    'XCART_OTHER_CSRF_STRATEGY',
    'XCART_OTHER_META_UPGRADE_INSECURE',
    'XCART_OTHER_SHOW_INITIALIZED_TRANSACTIONS',
    'XCART_OTHER_NEXT_PREVIOUS_ORDER_CRITERIA',
    'XCART_OTHER_COOKIE_SAME_SITE',
    'XCART_OTHER_COOKIE_HOST_ONLY',
];

$skipYamlConfig = [
    'host_details'       => true,
    'database_details'   => true,
    'cache'              => true,
    'log_details'        => true,
    'images'             => ['image_magick_path'],
    'installer_details'  => true,
    'storefront_options' => ['callback_opened', 'optimize_css'],
    'other'              => ['translation_driver', 'event_driver', 'use_sendfile', 'trusted_proxies'],
    'trial'              => ['end_date'],
    'demo'               => ['demo_mode'],
];

/**
 * Read XC configuration from config* files
 */
foreach ($configFiles as $configFileName) {
    $configFile = $configPath . '/' . $configFileName;

    if (is_readable($configFile)) {
        echo "Read data from {$configFile}\n";

        $data = parse_ini_file($configFile, true);

        if (is_array($data)) {
            /** @noinspection SlowArrayOperationsInLoopInspection */
            $configData = array_replace_recursive($configData, $data);
        }
    }
}

/**
 * Convert ini to env
 */

echo "Convert config data to env\n";

foreach ($configData as $section => $sectionData) {
    if (
        in_array(
            $section,
            [
                // unused
                'skin_details',
                'performance',
                'decorator',
                'service',
                'language',
                'installation',
                'amqp',
                'queue',
                'slapp',
            ],
            true)
    ) {
        continue;
    }

    if (
        in_array(
            $section,
            [
                'clean_urls',
                'clean_urls_aliases',
                'error_handling',
                'marketplace',
                'html_purifier',
                'html_purifier_additional_attributes',
                'export-import',
                'google_product_feed',
                'shipping_list',
            ],
            true)
    ) {
        foreach ($sectionData as $variable => $value) {
            $localYamlConfigData[$section][$variable] = $value;
        }

        continue;
    }

    foreach ($sectionData as $variable => $value) {
        $envData[x_name_convert($section . '_' . $variable)] = is_array($value) ? implode(',', $value) : $value;

        $localYamlConfigData[$section][$variable] = $value;
    }
}

/**
 * Get dist env data
 */

echo "Read dist env data\n";

$dotenv      = new Symfony\Component\Dotenv\Dotenv();
$distEnvData = $dotenv->parse(file_get_contents($distEnvFile));

/**
 * Generate env file content
 */

echo "Generate local env content (leave only changed values) \n";

$envData = x_convert_env_data($envData);

$envContent = '';
foreach ($envData as $variable => $value) {
    if (in_array($variable, $skipEnv, true)) {
        continue;
    }

    if (($distEnvData[$variable] ?? null) !== $value) {
        $value      = preg_match('/[#\s]/Ss', $value) ? "\"$value\"" : $value;
        $envContent .= $variable . '=' . $value . "\n";
    }
}

echo "Save data to file: $envFile \n";

file_put_contents($envFile, $envContent);

$distYamlConfigData = \Symfony\Component\Yaml\Yaml::parseFile($distYamlConfigFile);

foreach ($localYamlConfigData['clean_urls'] as $k => $v) {
    if (in_array($k, ['use_language_url', 'default_separator', 'capitalize_words', 'use_unicode'], true)) {
        continue;
    }

    $localYamlConfigData['clean_urls']['formats'][$k] = $v;
    unset($localYamlConfigData['clean_urls'][$k]);
}

$localYamlConfigData['clean_urls']['aliases'] = $localYamlConfigData['clean_urls_aliases'];
unset($localYamlConfigData['clean_urls_aliases']);

$htmlPurifierOptions = $localYamlConfigData['html_purifier'];
$htmlPurifierAttributes = $localYamlConfigData['html_purifier_additional_attributes'];
unset($localYamlConfigData['html_purifier_additional_attributes']);

$localYamlConfigData['html_purifier'] = [
    'options' => $htmlPurifierOptions,
    'attributes' => $htmlPurifierAttributes,
];

$localYamlConfigData['export_import'] = $localYamlConfigData['export-import'];
unset($localYamlConfigData['export-import']);

$yamlConfigContent = [];
foreach ($localYamlConfigData as $section => $sectionData) {
    if (($skipYamlConfig[$section] ?? null) === true) {
        continue;
    }

    foreach ($sectionData as $variable => $value) {
        if (in_array($variable, ($skipYamlConfig[$section] ?? []), true)) {
            continue;
        }

        $value = x_convert_yaml_value($section, $variable, $value);

        if (($distYamlConfigData['x_cart'][$section][$variable] ?? null) !== $value) {
            $yamlConfigContent[$section][$variable] = $value;
        }
    }
}

echo "Save data to file: $yamlConfigFile \n";

file_put_contents($yamlConfigFile, \Symfony\Component\Yaml\Yaml::dump(['x_cart' => $yamlConfigContent], 4, 3));

exit("Done\n");

/**
 * Display command usage help
 *
 * @return string
 */
function x_help(): string
{
    echo "Configuration converter\n";
    echo "This script converts configuration form etc/config*.php ini files to .env.local and config/local/parameters.yaml file\n";
    echo "\n";
    echo "Usage:\n";
    echo "    convert_config_to_env.php [options] [XC_root_path]\n";
    echo "\n";
    echo "Arguments:\n";
    echo "    XC_root_path    Path to the XC source code (default: __DIR__ . '/../../')";
    echo "\n";
    echo "Options:\n";
    echo "    -h              Display this help message\n";
    echo "\n";
    exit;
}

/**
 * Returns absolute real path by given relative path
 *
 * @param string $path
 *
 * @return string
 */
function x_get_absolute_path(string $path): string
{
    return realpath(strpos($path, '/') === 0 ? $path : (getcwd() . '/' . $path));
}

/**
 * @param string $name
 *
 * @return string
 */
function x_name_convert(string $name): string
{
    $name = ENV_PREFIX . strtoupper(str_replace(['-', '.'], '_', camelToUnderscore($name)));

    $nameFixes = [
        'XCART_OTHER_COOKIE_HOSTONLY' => 'XCART_OTHER_COOKIE_HOST_ONLY',
        'XCART_OTHER_COOKIE_SAMESITE' => 'XCART_OTHER_COOKIE_SAME_SITE',

        'XCART_DATABASE_DETAILS_TABLE_PREFIX' => 'DATABASE_DEFAULT_TABLE_PREFIX',
        'XCART_DATABASE_DETAILS_SOCKET'       => 'DATABASE_UNIX_SOCKET',
        'XCART_DATABASE_DETAILS_CHARSET'      => 'DATABASE_CHARSET',
    ];

    return $nameFixes[$name] ?? $name;
}

function x_convert_env_data(array $envData): array
{
    if (isset($envData['DATABASE_DEFAULT_TABLE_PREFIX'])) {
        $envData['DATABASE_DEFAULT_TABLE_PREFIX'] = rtrim($envData['DATABASE_DEFAULT_TABLE_PREFIX'], '_');
    }

    $envData['DATABASE_URL'] = constructDatabaseUrl([
        'host'     => $envData['XCART_DATABASE_DETAILS_HOSTSPEC'] ?? '',
        'port'     => $envData['XCART_DATABASE_DETAILS_PORT'] ?? '',
        'dbname'   => $envData['XCART_DATABASE_DETAILS_DATABASE'] ?? '',
        'user'     => $envData['XCART_DATABASE_DETAILS_USERNAME'] ?? '',
        'password' => $envData['XCART_DATABASE_DETAILS_PASSWORD'] ?? '',
    ]);

    return $envData;
}

function x_convert_yaml_value(string $section, string $variable, $value)
{
    $bool = [
        'clean_urls'    => ['use_language_url', 'capitalize_words', 'use_unicode'],
        'images'        => ['make_progressive', 'generate_retina_images'],
        'html_purifier' => ['Attr.EnableID', 'CSS.AllowTricky', 'HTML.SafeEmbed', 'HTML.SafeObject', 'HTML.SafeIframe'],
        'other'         => ['meta_upgrade_insecure', 'show_initialized_transactions', 'cookie_hostonly'],
    ];

    $int = [
        'marketplace'   => ['upgrade_step_time_limit'],
        'shipping_list' => ['display_selector_cutoff'],
    ];

    $array = [
        'storefront_options' => ['autocomplete_states_for_countries'],
        'other'              => ['trusted_domains', 'trusted_proxies'],
    ];

    if (in_array($variable, $bool[$section] ?? [], true)) {
        $value = strtolower($value);

        return !in_array($value, ['off', 'n'], true) && (bool) $value;
    }

    if (in_array($variable, $int[$section] ?? [], true)) {
        return (int) $value;
    }

    if (in_array($variable, $array[$section] ?? [], true)) {
        return array_filter(array_map('trim', explode(',', $value)));
    }

    return $value;
}

/**
 * @param $string
 * @param $us
 *
 * @return string
 * @see https://stackoverflow.com/a/40514305
 */
function camelToUnderscore($string, $us = '_')
{
    return strtolower(preg_replace(
        '/(?<=\d)(?=[A-Za-z])|(?<=[A-Za-z])(?=\d)|(?<=[a-z])(?=[A-Z])/', $us, $string));
}

/**
 * @param mixed[] $params
 */
function constructDatabaseUrl(array $params): string
{
    $url = '//';

    if (($params['user'] ?? '')) {
        $url .= $params['user'];
        if (($params['password'] ?? '')) {
            $url .= ':' . $params['password'];
        }

        $url .= '@';
    }

    $url .= empty($params['host']) ? 'localhost' : $params['host'];

    if (!empty($params['port'])) {
        $url .= ':' . $params['port'];
    }

    if (!empty($params['dbname'])) {
        $url .= '/' . $params['dbname'];
    }

    $url .= '?serverVersion=5.7';

    return $url;
}
