/* vim: set ts=2 sw=2 sts=2 et: */

/**
 * Upgrade
 *
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

const currentUrl = new URL(window.location.href);
const params = currentUrl.searchParams;

if (params.get('xcUrl')) {
  sessionStorage.setItem('xcUrl', params.get('xcUrl'));
}

const xcUrl = sessionStorage.getItem('xcUrl') || '';
const scenarioId = params.get('scenarioId');
const returnURL = params.get('returnURL');
const isPostUpgrade = !!params.get('isPostUpgrade');
const isUpgradeTo55 = !!params.get('isUpgradeTo55');

const steps = [
  {
    type: 'PUT',
    url: `${xcUrl}/service.php/api/scenarios/upgrade/${scenarioId}/upgrade_entries`,
    progressMessage: 'Set upgrade entries',
    isIterativeStep: false,
    skip: isUpgradeTo55,
  },
  {
    type: 'GET',
    url: `${xcUrl}/service.php/api/incorrect_permissions_commands?scenarioId=${scenarioId}`,
    progressMessage: 'Check permissions',
    callback: 'showPermissionPopup',
    isIterativeStep: false,
    skip: isUpgradeTo55,
  },
  {
    type: 'GET',
    url: `${xcUrl}/service.php/api/integrity_violations?scenarioId=${scenarioId}`,
    progressMessage: 'Check integrity violations',
    callback: 'showIntegrityViolationsPopup',
    isIterativeStep: false,
    skip: isUpgradeTo55,
  },
  {
    type: 'GET',
    url: `${xcUrl}/service.php/api/upgrade_notes/pre?scenarioId=${scenarioId}`,
    progressMessage: 'Check pre-upgrade notes',
    callback: 'showPreUpgradeNotesPopup',
    isIterativeStep: false,
    skip: isUpgradeTo55,
  },
  {
    type: 'PUT',
    url: `${xcUrl}/service.php/api/scenarios/upgrade/${scenarioId}/download_packs`,
    progressMessage: 'Download packs',
    isIterativeStep: true,
    skip: isUpgradeTo55,
  },
  {
    type: 'PUT',
    url: `${xcUrl}/service.php/api/scenarios/upgrade/${scenarioId}/start`,
    progressMessage: 'Start scenario',
    isIterativeStep: false,
    skip: false,
  },
  {
    type: 'GET',
    url: `${xcUrl}/service.php/api/upgrade_notes/post?scenarioId=${scenarioId}`,
    progressMessage: 'Check post-upgrade notes',
    callback: 'showPostUpgradeNotesPopup',
    isIterativeStep: false,
    skip: false,
  },
];

const stepsCount = steps.length;
let currentStep = 0;
let progressMessages = [];

if (isPostUpgrade) {
  currentStep = 6;
  steps.forEach(step => progressMessages.push(step.progressMessage));
}

callSteps();

function callSteps() {
  if (!scenarioId) {
    $('.title').html('Error. ScenarioId is empty');

    return;
  }

  if (!returnURL) {
    $('.title').html('Error. ReturnURL is empty');

    return;
  }

  runNextStep();
}

function runNextStep() {
  if (!isPostUpgrade && currentStep === 6) {
    // if the rebuild has not yet been completed and the current step is post_upgrade_notes
    // then redirect to the rebuild page
    currentUrl.searchParams.append('isUpgrade', '1');
    window.location.href = currentUrl.href.replace('upgrade.html', 'rebuild.html');
  } else if (currentStep > 6) {
    // if all steps have been completed (last step index is 5)
    // then redirect to the returnURL or success on update
    window.location.href = isUpgradeTo55 ? returnURL : `${xcUrl}/admin/?target=apps#/success-rebuild?`;
  } else {
    // run next step
    const step = steps[currentStep];

    if (step.skip) {
      currentStep++;
      runNextStep();
    } else {
      $.ajax({
        method: step.type,
        contentType: 'application/json',
        dataType: 'json',
        timeout: 0,
        url: step.url,
        data: JSON.stringify({}),
        success: function (data, textStatus, xhr) {
          const shouldRepeatCurrentStep = (step.isIterativeStep && data.status === 'inProgress');

          progressMessages[currentStep] = shouldRepeatCurrentStep
            ? data.message
            : step.progressMessage;

          $('.rebuild-starting').hide();
          $('.rebuild-started').show();

          refreshProgressStatus();

          if (data.length && step.callback) {
            window[step.callback](data);
          } else {
            if (!shouldRepeatCurrentStep) {
              currentStep++;
            }

            runNextStep();
          }
        },
        error: function (xhr, textStatus, errorThrown) {
          $('.rebuild-starting').hide();
          $('.rebuild-started').hide();
          $('.spinner-ring').hide();

          let errorText = `<span class="error">${xhr.status} ${errorThrown}</span>`;

          if (xhr.responseJSON) {
            const detailedMessage = xhr.responseJSON.detail || xhr.responseJSON.message;

            if (detailedMessage === errorThrown) {
              errorText += `<span class="error">Error details are available in <code>var/log</code></span>`;
              errorText += `<span class="error">Otherwise you can view detailed error information by enabling debug mode (<code>APP_DEBUG=true</code>) and running the update again.</span>`;
            } else {
              errorText += `<span class="error">${xhr.responseJSON.detail || xhr.responseJSON.message}</span>`;
            }
          } else if (xhr.responseText) {
            errorText += `<span class="error">${xhr.responseText}</span>`;
          }

          $('.rebuild-error')
            .prepend(errorText)
            .show();
          $('#storefront').attr('href', xcUrl + '/');
          $('#dashboard').attr('href', xcUrl + '/admin/');
        },
      });
    }
  }
}

function refreshProgressStatus() {
  let progressStatus = [];
  let stepOfSteps = '';

  for (let i = 0; i < progressMessages.length; i++) {
    const message = progressMessages[i];

    const progressStep = `<li class="progress-message"><span class="common">${message}</span></li>`;

    progressStatus.unshift(progressStep);

    const step = i + 1;
    stepOfSteps = step + ' of ' + stepsCount;

    $('.progress-inner').css({
      'width': 100 / stepsCount * step + '%',
    });
  }

  $('#step-of-steps').html(stepOfSteps);
  $('.progress-status').html(progressStatus.join(''));
}

function showPermissionPopup(data) {
  let dialogText =
    '<div>' +
    '<p>Not enough permissions to run the process. Make sure the following permissions are set (UNIX-like systems only).</p>' +
    '<p>Such permissions are required for a seamless automated installation or upgrade of X-Cart on your server. ' +
    'They do not take into account the specific configuration of your server or any security requirements. ' +
    'Once the process is completed, make sure you change the permissions to a more restrictive setting. ' +
    '<a target=\'_blank\' href=\'https://kb.x-cart.com/en/setting_up_x-cart_5_environment/secure_configuration.html#why-x-cart-asks-for-666777-permissions\'>Read more</a>' +
    '</p>' +
    '</div>' +
    '<code class="permissions-command">';

  data.forEach(incorrectPermissionsCommand => dialogText += `${incorrectPermissionsCommand.cmd}<br>`);

  dialogText += '</code>';


  $('#dialog')
    .html(dialogText)
    .dialog({
      modal: true,
      title: 'Permissions',
      buttons: [
        {
          class: 'copy2clipboard button',
          text: 'Copy to clipboard',
          click: function () {

          },
        },
        {
          class: 'right button',
          text: 'Cancel',
          click: function () {
            window.location.href = returnURL;
          },
        },
        {
          class: 'regular-main-button button',
          text: 'Retry',
          click: function () {
            $(this).dialog('close');
            runNextStep();
          },
        },
      ],
    });

  const clipboard = new ClipboardJS(
    '.copy2clipboard',
    {
      target: function () {
        return $('.permissions-command')[0];
      },
    },
  );

  clipboard.on('success', function (e) {
    alert('Copied to clipboard');

    e.clearSelection();
  });
}

function showIntegrityViolationsPopup(data) {
  let dialogText =
    '<div>' +
    'The system has detected some custom modifications have been implemented ' +
    'on your X-Cart installation by direct modification of the core/module ' +
    'files instead of by creating a module modifying the original functionality.' +
    '</div>' +
    '<div class="violations">';

  data.forEach(integrityViolation => dialogText += `<div>${integrityViolation.filePath}</div>`);

  dialogText += '</div>';


  $('#dialog')
    .html(dialogText)
    .dialog({
      modal: true,
      title: 'Some files have been modified',
      buttons: [
        {
          class: 'right button',
          text: 'Cancel',
          click: function () {
            window.location.href = returnURL;
          },
        },
        {
          class: 'regular-main-button button',
          text: 'Continue',
          click: function () {
            currentStep++;
            $(this).dialog('close');
            runNextStep();
          },
        },
      ],
    });
}

function showPreUpgradeNotesPopup(data) {
  let dialogText = '';

  data.forEach(upgradeNote => dialogText += getUpgradeNoteItem(upgradeNote));

  $('#dialog')
    .html(dialogText)
    .dialog({
      modal: true,
      title: 'Pre upgrade notes',
      buttons: [
        {
          class: 'button',
          text: 'Cancel',
          click: function () {
            window.location.href = returnURL;
          },
        },
        {
          class: 'regular-main-button button',
          text: 'Continue',
          click: function () {
            currentStep++;
            $(this).dialog('close');
            runNextStep();
          },
        },
      ],
    });
}

function showPostUpgradeNotesPopup(data) {
  let dialogText = '';

  data.forEach(upgradeNote => dialogText += getUpgradeNoteItem(upgradeNote));

  $('#dialog')
    .html(dialogText)
    .dialog({
      modal: true,
      title: 'Post upgrade notes',
      buttons: [
        {
          class: 'button',
          text: 'Continue',
          click: function () {
            currentStep++;
            $(this).dialog('close');
            runNextStep();
          },
        },
      ],
    });
}

function isHTMLString(string) {
  return /<\/?[a-z][\s\S]*>/i.test(string);
}

function getUpgradeNoteItem(upgradeNote) {
  let result = '<div class="upgrade-note">';
  result += `<h3 class="upgrade-note-title">${upgradeNote.moduleName} (${upgradeNote.authorName})</h3>`;

  let parsedUpgradeNotes = [];

  upgradeNote.notes.forEach(note => {
    if (isHTMLString(note)) {
      parsedUpgradeNotes.push(note);
    } else {
      parsedUpgradeNotes = parsedUpgradeNotes.concat(
        note.split('\n').filter(line => !!line)
      );
    }
  });

  parsedUpgradeNotes.forEach(note => result += `<p>${note}</p>`);

  return result;
}
